/**
******************************************************************************
* @file    CANReceive.c
* @project CAN-ENET Software Support Package
*
* @version V4.0.0
* @date    April 2025
* @author  Oleksandr Bogush
*
* @copyright
* (c) Axiomatic Technologies Corp. All rights reserved.
*
* @brief
* This is an example console application showing how CAN frames can be received
* from the Axiomatic Ethernet to CAN converter.
*
******************************************************************************
* @attention
*
* <h2><center>&copy; COPYRIGHT(c) 2025 Axiomatic Technologies Corporation</center></h2>
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
*      may be used to endorse or promote products derived from this software
*      without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************
*/
/// @cond
#if defined(_WIN32) || defined(_WIN64) || defined(DOXYGEN)
 #define WINDOWS 1
#else
 #define WINDOWS 0
#endif

#ifndef WINDOWS
 #error Macro WINDOWS should be defined. WINDOWS={0-Not running on Windows, 1-Running on Windows}
#endif

#if WINDOWS==1
  #define _CRT_SECURE_NO_WARNINGS
  #define _WINSOCK_DEPRECATED_NO_WARNINGS

  #include <winsock2.h>
  #include <ws2tcpip.h>
  #pragma comment(lib, "Ws2_32.lib")

  #define SocketCleanup() WSACleanup()
  #define GetSocketError() WSAGetLastError()
  #define close closesocket

  #define MS_PER_TICK (1000.0 / CLOCKS_PER_SEC)
#else
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <sys/socket.h>
  #include <arpa/inet.h>
  #include <errno.h>
  #include <unistd.h>

  #define SocketCleanup()
  #define GetSocketError() errno

  static double GetTimeDiffInMS(const struct timespec *pTimeStop,const struct timespec *pTimeStart);
#endif

#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <time.h>

#include "../Inc/CommProtocol.h"

#define SOCKET_TIMEOUT 100 /* [ms] */
#define SOCKET_RECEIVING_TIME 10 /* [s] */

static void OnDataParsed(PM_PROTOCOL_MESSAGE_t *pPMessage, void *arg);
static void OnDataParsedError(void *arg);

static void OnCANFrameParsed(CP_CAN_FRAME_t *pCANFrame, void *arg); // Obsolete
static void OnCANFDFrameParsed(CP_CAN_FD_FRAME_t* pCANFDFrame, CP_CAN_FRAME_ROUTING_DATA_t* pCANFrameRoutingData, DWORD_t dwAbsTimeStamp, void* arg);

typedef enum
{
    UDP = 0,
    TCP
}
PORT_TYPE;

typedef struct
{
    int iSocket;
    int iNumberOfCANFrames;
    DWORD dwTimeStampOld;

    BOOL bIsProgramExitRequested;
}
PARSING_ARG;

int main( int argc, const char* argv[] )
{
    char szDeviceAddress[256];
    int iResult;
    unsigned int uPort;
    unsigned int uPortType;
    PORT_TYPE PortType;
    int iSocket;
    struct sockaddr_in SocketAddr;
    BYTE_t RxData[PM_PROTOCOL_MESSAGE_BUFFER_SIZE];
    PM_PROTOCOL_PARSER_t PParser;
    int iBytesReceived;
    BYTE_t TxData[PM_PROTOCOL_MESSAGE_BUFFER_SIZE];
    PM_PROTOCOL_MESSAGE_t PMessage;
    BOOL_t bResult;
    int iBytesToSend;
    int iBytesSent;
    PARSING_ARG ParsingArg;
    unsigned int uContinue;

    #if WINDOWS==1
      WSADATA wsaData;
      BOOL bOptionValue;
      DWORD dwOptionValue;
      clock_t ClockStartTime;
      clock_t ClockHeartbeatStartTime;
    #else
      int iOptionValue;
      struct timeval TimeoutOptionValue;
      struct timespec ClockStartTime,ClockHeartbeatStartTime,ClockTime;
    #endif

    printf ("===============================================================\n"
            " Program: CANReceive V4.0.0\n"
            " CAN-ENET Software Support Package. p/n AX140910\n"
            " (c) Axiomatic Technologies Corporation\n\n"
            " This is an example program showing how CAN frames can be received\n"
            " from Axiomatic Ethernet to CAN converter.\n"
            "===============================================================\n");

    #if WINDOWS==1
      /* We need to initialize Winsock if running on Windows*/
      iResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
      if (iResult != 0)
      {
          printf("WSAStartup() failed with error: %d\n", iResult);
          return 1;
      }
    #endif

    /* First we should establish a socket for the IP communication with the converter (server).
       We need the Device IP address, Port Number and the Port Type */
    printf("Enter the Device IP address (xxx.xxx.xxx.xxx) : ");
    scanf("%s", szDeviceAddress);
    printf("Enter the Port Number : ");
    scanf("%u", &uPort);
    printf("Enter the Port Type (0-UDP, 1-TCP) : ");
    scanf("%u", &uPortType);
    PortType = uPortType == 0 ? UDP : TCP;

    /* Creating socket for connecting to the server */
    iSocket = PortType == UDP ? socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP) : socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (iSocket < 0)
    {
        printf("socket() failed with error code: %d\n", GetSocketError());
        SocketCleanup();
        return 1;
    }

    /* If TCP protocol is used, set the TCP_NODELAY option to avoid delays in sending protocol messages */
    if (PortType == TCP)
    {
        /* Set the TCP_NODELAY socket option */
        #if  WINDOWS==1
          bOptionValue = TRUE;
          iResult = setsockopt(iSocket, IPPROTO_TCP, TCP_NODELAY, (char*)&bOptionValue, sizeof(bOptionValue));
        #else
          iOptionValue = 1;
          iResult = setsockopt(iSocket, IPPROTO_TCP, TCP_NODELAY, &iOptionValue, sizeof(iOptionValue));
        #endif
        if(iResult<0)
        {
            printf("setsockopt() failed with error code: %d\n", GetSocketError());
            close(iSocket);
            SocketCleanup();
            return 1;
        }
    }

    /* Set socket receiving operation timeout. This timeout is necessary in case the converter stops sending any messages */
    #if  WINDOWS==1
      dwOptionValue = SOCKET_TIMEOUT;
      iResult = setsockopt(iSocket, SOL_SOCKET, SO_RCVTIMEO, (char*)&dwOptionValue, sizeof(dwOptionValue));
    #else
      memset(&TimeoutOptionValue, 0, sizeof(TimeoutOptionValue));
      TimeoutOptionValue.tv_sec = SOCKET_TIMEOUT / 1000;
      TimeoutOptionValue.tv_usec = (SOCKET_TIMEOUT - TimeoutOptionValue.tv_sec * 1000) * 1000;
      iResult = setsockopt(iSocket, SOL_SOCKET, SO_RCVTIMEO, &TimeoutOptionValue, sizeof(TimeoutOptionValue));
    #endif
    if (iResult < 0)
    {
        printf("setsockopt() failed with error code: %d\n", GetSocketError());
        close(iSocket);
        SocketCleanup();
        return 1;
    }

    /* Preparing the server address data*/
    memset(&SocketAddr, 0, sizeof(SocketAddr));
    SocketAddr.sin_family = AF_INET;
    SocketAddr.sin_addr.s_addr = inet_addr(szDeviceAddress);
    SocketAddr.sin_port = htons(uPort);

    /* Connecting to the server */
    iResult = connect(iSocket,(struct sockaddr *)&SocketAddr,sizeof(SocketAddr));
    if (iResult != 0)
    {
        printf("connect() failed with error code: %d\n", GetSocketError());
        close(iSocket);
        SocketCleanup();
        return 1;
    }

    /* Receiving the data */
    do
    {
        /* Initializing the parser */
        memset(&PParser, 0, sizeof(PParser));

        /* Initializing the timing variables. Since POSIX requires that the CLOCKS_PER_SEC value
           be one million independent of the actual resolution, the clock() function cannot be used
           to accurately measure time intervals in Linux. We will use clock_gettime() POSIX function
           instead. */
        #if WINDOWS==1
          ClockStartTime = clock();
        #else
          clock_gettime(CLOCK_MONOTONIC,&ClockStartTime);
        #endif
        ClockHeartbeatStartTime = ClockStartTime;

        /* If UDP protocol is used, the client should let the server know its socket address
           by sending a message to the server. We will use a Heartbeat message. */
        if (PortType == UDP)
        {
            /* Preparing the Heartbeat message. */
            CPGenHearbeatMessage(&PMessage, 0, 0, 0, (CP_CONVERTER_TYPE_t)0, CP_SUPPORTED_FEATURE_FLAG_CAN_FD_STREAM,NULL);

            /* Copying the message to the transmit buffer TxData. */
            bResult = PMCopyToBuffer(&PMessage, TxData, sizeof(TxData), &iBytesToSend);
            assert(bResult);

            /* Sending the Heartbeat message. */
            iBytesSent = send(iSocket, TxData, iBytesToSend, 0);
            if (iBytesSent < 0)
            {
                printf("send() failed with error code: %d\n", GetSocketError());
                close(iSocket);
                SocketCleanup();
                return 1;
            }
        }
        /* Prepare parsing data */
        ParsingArg.iNumberOfCANFrames = 0;
        ParsingArg.iSocket = iSocket;
        ParsingArg.bIsProgramExitRequested = FALSE;

        /* Receiving data in the RxData buffer. */
        do
        {
            iBytesReceived = recv(iSocket, RxData, sizeof(RxData), 0);
            if (iBytesReceived > 0)
            {
                /* Data has been successfully received.
                   Now we are calling the protocol message parser. */
                PMParseFromBuffer(RxData, iBytesReceived, &PParser, OnDataParsed, OnDataParsedError, &ParsingArg);

                if (ParsingArg.bIsProgramExitRequested) return 1;
            }
            else if (iBytesReceived == 0)
            {
                /* Connection has been closed on the server side */
                printf("Connection has been closed on the server side.\n");
                close(iSocket);
                SocketCleanup();
                return 1;
            }
            else
            {
                iResult = GetSocketError();
                #if  WINDOWS==1
                  if (iResult != WSAETIMEDOUT)
                #else
                  if (iResult != EAGAIN && iResult != EWOULDBLOCK && iResult != EINPROGRESS)
                #endif
                {
                    /* Receiving error */
                    printf("recv() failed with error: %d\n", iResult);
                    close(iSocket);
                    SocketCleanup();
                    return 1;
                }
            }
            if (PortType == UDP)
            {
                /* You need to maintain connection to avoid disconnection on inactivity */
                #if WINDOWS==1
                  if ((clock() - ClockHeartbeatStartTime)*MS_PER_TICK >= CP_HEARTBEAT_MESSAGE_INTERVAL)
                #else
                clock_gettime(CLOCK_MONOTONIC,&ClockTime);
                  if(GetTimeDiffInMS(&ClockTime,&ClockHeartbeatStartTime)>=CP_HEARTBEAT_MESSAGE_INTERVAL)
                #endif
                {
                    /* Sending a Heartbeat message */
                    #if WINDOWS==1
                      ClockHeartbeatStartTime = clock();
                    #else
                      clock_gettime(CLOCK_MONOTONIC,&ClockHeartbeatStartTime);
                    #endif

                    /* Preparing the Heartbeat message. */
                    CPGenHearbeatMessage(&PMessage, 0, 0, 0, (CP_CONVERTER_TYPE_t)0, CP_SUPPORTED_FEATURE_FLAG_CAN_FD_STREAM, NULL);

                    /* Copying the message to the transmit buffer TxData. */
                    bResult = PMCopyToBuffer(&PMessage, TxData, sizeof(TxData), &iBytesToSend);
                    assert(bResult);

                    /* Sending the Heartbeat message. */
                    iBytesSent = send(iSocket, TxData, iBytesToSend, 0);
                    if (iBytesSent < 0)
                    {
                        printf("send() failed with error code: %d\n", GetSocketError());
                        close(iSocket);
                        SocketCleanup();
                        return 1;
                    }
                }
            }

        #if WINDOWS==1
        } while (((clock() - ClockStartTime)*MS_PER_TICK) < (SOCKET_RECEIVING_TIME * 1000));
        #else
        clock_gettime(CLOCK_MONOTONIC,&ClockTime);
        } while(GetTimeDiffInMS(&ClockTime,&ClockStartTime)/1000< SOCKET_RECEIVING_TIME);
        #endif

        printf("Number of the received CAN frames : %d\n", ParsingArg.iNumberOfCANFrames);
        printf("Would you like to continue (1 - Yes, 0 - No) : ");
        scanf("%u", &uContinue);
    }
    while (uContinue != 0);

    printf("Good bye.\n");
    close(iSocket);
    SocketCleanup();

    return 0;
}

/**
  * @brief  This callback function is called on successful parsing of the protocol message.
  *
  * @param[in]  pPMessage: pointer to the parsed protocol message.
  * @param[in]  arg: pointer to an arbitrary data passed to PMParseFromBuffer() function.
  *
  * @retval None
  */
void OnDataParsed(PM_PROTOCOL_MESSAGE_t *pPMessage, void *arg)
{
    PM_PROTOCOL_MESSAGE_t PMessage;
    BOOL_t bResult;
    int iBytesToSend;
    int iBytesSent;
    BYTE_t TxData[PM_PROTOCOL_MESSAGE_BUFFER_SIZE];
    PARSING_ARG* ParsingArg = (PARSING_ARG*)arg;
   
    /* Parsing Communication Protocol Messages. */

    /* Parsing Status Request. */
    if (CPParseStatusRequest(pPMessage))
    {
        /* You need to respond with a feature flag CP_SUPPORTED_FEATURE_FLAG_CAN_FD_STREAM to enable CAN FD messages.
           If you want to use the old CAN and Notification stream format with a server supporting CAN FD, you still need to
           respond to the Status Request message or generate at least one Heartbeat message to let the server know what
           CAN format should be used. Any old Status Response or Heartbeat message format can be used, which will trigger
           the old CAN and Notification stream format for the server-client communication.*/
        CPGenStatusResponseMessage(&PMessage, 0, 0, 0, 0, (CP_CONVERTER_TYPE_t)0, CP_SUPPORTED_FEATURE_FLAG_CAN_FD_STREAM, NULL);

        /* Copying the message to the transmit buffer TxData. */
        bResult = PMCopyToBuffer(&PMessage, TxData, sizeof(TxData), &iBytesToSend);
        assert(bResult);

        /* Sending the Status Response  message. */
        iBytesSent = send(ParsingArg->iSocket, TxData, iBytesToSend, 0);
        if (iBytesSent < 0)
        {
            printf("send() failed with error code: %d\n", GetSocketError());
            close(ParsingArg->iSocket);
            SocketCleanup();
            ParsingArg->bIsProgramExitRequested = TRUE;
        }
        return;
    }

    /* Parsing CAN FD Stream. */
    if (CPParseCANFDStream(pPMessage, OnCANFDFrameParsed, arg))
    {
        /* The CAN FD Stream has been parsed. Add your code here if necessary. */
        return;
    }

    /* Parsing CAN and Notification Stream. Obsolete */
    if (CPParseCANDataAndNotificationStream(pPMessage, OnCANFrameParsed, NULL, arg))
    {
        /* The CAN and Notification Stream has been parsed. Add your code here if necessary. */
        return;
    }
}

/**
  * @brief  This callback function is called on the protocol message parsing error.
  *
  * @param[in]  arg: pointer to an arbitrary data passed to PMParseFromBuffer() function.
  *
  * @retval None
  */
void OnDataParsedError(void *arg)
{
    printf("PMessage parsing error.\n");
}

/**
  * @brief  This callback function is called on successful parsing of the CAN frame. Obsolete
  *
  * @param[in]  pCANFrame: pointer to the parsed CAN frame.
  * @param[in]  arg: pointer to an arbitrary data passed to CPParseCANDataAndNotificationStream()
  *                  function.
  *
  * @retval None
  */
void OnCANFrameParsed(CP_CAN_FRAME_t *pCANFrame, void *arg)
{
    int i;
    PARSING_ARG* ParsingArg = (PARSING_ARG*)arg;

    printf("CANFrame (Legacy Format) #%4d: TS=%5lu Id=%8.8lX E=%1u R=%1u L=%1d", ++ParsingArg->iNumberOfCANFrames, (long unsigned int)pCANFrame->dwTimeStamp, (long unsigned int)pCANFrame->dwId,
            pCANFrame->bEIdFlag, pCANFrame->bRemFrameFlag, pCANFrame->bLength);
    if (pCANFrame->bRemFrameFlag)
    {
        printf("\n");
    }
    else
    {
        printf("   ");
        for (i = 0; i < pCANFrame->bLength; i++)
        {
            printf("D%1d=%2.2X ", i,pCANFrame->Data[i]);
        }
        printf("\n");
    }
}

/**
  * @brief  This callback function is called on successful parsing of CAN FD frame.
  *
  * @param[in]  pCANFDFrame: pointer to the parsed CAN FD frame.
  * @param[in]  pCANFrameRoutingData: pointer to the CAN FD frame routing data.
  * @param[in]  dwAbsTimeStamp: Absolute timestamp in ms.
  * @param[in]  arg: pointer to an arbitrary data passed to CPParseCANFDStream()
  *                  function.
  *
  *	@retval None
  */
void OnCANFDFrameParsed(CP_CAN_FD_FRAME_t* pCANFDFrame, CP_CAN_FRAME_ROUTING_DATA_t* pCANFrameRoutingData, DWORD_t dwAbsTimeStamp, void* arg)
{
    int i;
    PARSING_ARG* ParsingArg = (PARSING_ARG*)arg;
    DWORD_t dwRelTimeStamp;

    if (ParsingArg->iNumberOfCANFrames == 0) ParsingArg->dwTimeStampOld = dwAbsTimeStamp;
    dwRelTimeStamp = (dwAbsTimeStamp - ParsingArg->dwTimeStampOld);
    ParsingArg->dwTimeStampOld = dwAbsTimeStamp;

    if ((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_ERROR_MESSAGE) > 0)
    {
        printf("CANFrame (CAN FD Frame Format) #%4d: Error message L=%2d", ++ParsingArg->iNumberOfCANFrames, pCANFDFrame->bLength);
        printf("   ");
        for (i = 0; i < pCANFDFrame->bLength; i++)
        {
            printf("D%1d=%2.2X ", i, pCANFDFrame->Data[i]);
        }
        printf("\n");
        return;
    }
    printf("CANFrame (CAN FD Frame Format) #%4d: TS=%5lu Id=%8.8lX E=%1u R=%1u FD=%1d FD_BRS=%1d FD_ESI=%1d L=%2d", ++ParsingArg->iNumberOfCANFrames, (long unsigned int)dwRelTimeStamp, (long unsigned int)pCANFDFrame->dwID,
                   (long unsigned int)((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_EXTENDED_ID)>0),
                   (long unsigned int)((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_REMOTE_FRAME) > 0),
                   (long unsigned int)((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_FD_FRAME) > 0),
                   (long unsigned int)((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_FD_BRS) > 0),
                   (long unsigned int)((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_FD_ESI) > 0),
                   pCANFDFrame->bLength);
    if (((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_FD_FRAME)>0) || ((pCANFDFrame->bFlags & CP_CAN_FD_FLAG_REMOTE_FRAME) == 0))
    {
        printf("   ");
        for (i = 0; i < pCANFDFrame->bLength; i++)
        {
            printf("D%1d=%2.2X ", i, pCANFDFrame->Data[i]);
        }
        printf("\n");
    }
    else
    {
        printf("   ");
    }
}

#if WINDOWS==0
/**
  * @brief  Calculates time difference in milliseconds between two time stamps
  *         presented by structure timespec in POSIX OS.
  *
  * @param[in]  pTimeStop: pointer to the time stop time stamp.
  * @param[in]  pTimeStart: pointer to the time start time stamp.
  *
  * @retval Time in milliseconds between time start and time stop time stamps.
  */
double GetTimeDiffInMS(const struct timespec *pTimeStop,const struct timespec *pTimeStart)
{
    long lTimeDiffInNS;
    double dTimeDiffInS;

    lTimeDiffInNS=pTimeStop->tv_nsec-pTimeStart->tv_nsec;
    dTimeDiffInS=difftime(pTimeStop->tv_sec,pTimeStart->tv_sec);
    if(lTimeDiffInNS<0)
    {
        lTimeDiffInNS+=1000000000L;
        dTimeDiffInS-=1;
    }
    return dTimeDiffInS*1000.0+((double)lTimeDiffInNS)/1000000.0;
}
#endif
/// @endcond
