/**
******************************************************************************
* @file    CANSend.c
* @project CAN-ENET Software Support Package
*
* @version V3.0.0
* @date    December 2022
* @author  Oleksandr Bogush
*
* @copyright
* (c) Axiomatic Technologies Corp. All rights reserved.
*
* @brief
* This is an example console application showing how CAN frames can be sent to
* the Axiomatic Ethernet to CAN converter.
*
******************************************************************************
* @attention
*
* <h2><center>&copy; COPYRIGHT(c) 2022 Axiomatic Technologies Corporation</center></h2>
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
*      may be used to endorse or promote products derived from this software
*      without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************
*/
/// @cond
#if defined(_WIN32) || defined(_WIN64) || defined(DOXYGEN)
 #define WINDOWS 1
#else
 #define WINDOWS 0
#endif

#ifndef WINDOWS
 #error Macro WINDOWS should be defined. WINDOWS={0-Not running on Windows, 1-Running on Windows}
#endif

#if WINDOWS==1
  #define _CRT_SECURE_NO_WARNINGS
  #define _WINSOCK_DEPRECATED_NO_WARNINGS

  #include <winsock2.h>
  #include <ws2tcpip.h>
  #pragma comment(lib, "Ws2_32.lib")

  #define SocketCleanup() WSACleanup()
  #define GetSocketError() WSAGetLastError()
  #define close closesocket
#else
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <sys/socket.h>
  #include <arpa/inet.h>
  #include <errno.h>
  #include <unistd.h>

  #define SocketCleanup()
  #define GetSocketError() errno
#endif

#include <stdio.h>
#include <string.h>
#include <assert.h>

#include "../Inc/CommProtocol.h"

#define STANDARD_CAN_ID_MAX 0x7ff
#define EXTENDED_CAN_ID_MAX 0x1fffffff

#define CAN_DATA_MAX 0xff

typedef enum
{
    UDP = 0,
    TCP
}
PORT_TYPE;

const CP_CAN_FRAME_ROUTING_DATA_t gcCANFrameRoutingData = { 0, 0, CP_CAN_AND_NOTIFICATION_STREAM_GROUP, CP_CAN_AND_NOTIFICATION_STREAM_CHANNEL_ID_SET };

int main( int argc, const char* argv[] )
{
    char szDeviceAddress[256];
    int iResult;
    unsigned int uPort;
    unsigned int uPortType;
    PORT_TYPE PortType;
    int iSocket;
    struct sockaddr_in SocketAddr;
    BYTE_t TxData[PM_PROTOCOL_MESSAGE_BUFFER_SIZE];
    PM_PROTOCOL_MESSAGE_t PMessage;
    BOOL_t bResult;
    int iBytesToSend;
    int iBytesSent;
    int i,j;

    unsigned long ulCANId;
    CP_CAN_FRAME_t CANFrame;
    CP_CAN_FD_FRAME_t CANFDFrame;
    unsigned int uIsCANFDStreamFormat;
    unsigned int uIsCANFDFrame;
    unsigned int uIsCANFD_BRS;
    unsigned int uIsCANFrameExtended;
    unsigned int uCANDataLength;
    unsigned int uIsCANRemFrameRequest;
    unsigned int CANData[CP_CAN_FD_MAX_DATA_LENGTH];
    unsigned int uCANDataLengthMax;
    int iNumberOfMessages;
    unsigned int uContinue;

    #if WINDOWS==1
      WSADATA wsaData;
      BOOL bOptionValue;
    #else
      int iOptionValue;
    #endif

    printf ("===============================================================\n"
            " Program: CANSend V3.0.0\n"
            " CAN-ENET Software Support Package. p/n AX140910\n"
            " (c) Axiomatic Technologies Corporation\n\n"
            " This is an example showing how CAN frames can be sent\n"
            " to Axiomatic Ethernet to CAN converter.\n"
            "===============================================================\n");

    #if WINDOWS==1
      /* We need to initialize Winsock if running on Windows*/
      iResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
      if (iResult != 0)
      {
          printf("WSAStartup() failed with error: %d\n", iResult);
          return 1;
      }
    #endif

    /* First we should establish a socket for the IP communication with the converter (server).
       We need the Device IP address, Port Number and the Port Type */
    printf("Enter the Device IP address (xxx.xxx.xxx.xxx) : ");
    scanf("%s", szDeviceAddress);
    printf("Enter the Port Number : ");
    scanf("%u", &uPort);
    printf("Enter the Port Type (0-UDP, 1-TCP) : ");
    scanf("%u", &uPortType);
    PortType = uPortType == 0 ? UDP : TCP;

    /* Creating socket for connecting to the server */
    iSocket = PortType == UDP ? socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP) : socket(AF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (iSocket < 0)
    {
        printf("socket() failed with error code: %d\n", GetSocketError());
        SocketCleanup();
        return 1;
    }

    /* If TCP protocol is used, set the TCP_NODELAY option to avoid delays in sending protocol messages */
    if (PortType == TCP)
    {
        /* Set the TCP_NODELAY socket option */
        #if  WINDOWS==1
          bOptionValue = TRUE;
          iResult = setsockopt(iSocket, IPPROTO_TCP, TCP_NODELAY, (char*)&bOptionValue, sizeof(bOptionValue));
        #else
          iOptionValue = 1;
          iResult = setsockopt(iSocket, IPPROTO_TCP, TCP_NODELAY, &iOptionValue, sizeof(iOptionValue));
        #endif
        if(iResult<0)
        {
            printf("setsockopt() failed with error code: %d\n", GetSocketError());
            close(iSocket);
            SocketCleanup();
            return 1;
        }
    }

    /* Preparing the server address data*/
    memset(&SocketAddr, 0, sizeof(SocketAddr));
    SocketAddr.sin_family = AF_INET;
    SocketAddr.sin_addr.s_addr = inet_addr(szDeviceAddress);
    SocketAddr.sin_port = htons(uPort);

    /* Connecting to the server */
    iResult = connect(iSocket,(struct sockaddr *)&SocketAddr,sizeof(SocketAddr));
    if (iResult != 0)
    {
        printf("connect() failed with error code: %d\n", GetSocketError());
        close(iSocket);
        SocketCleanup();
        return 1;
    }

    do
    {
        do
        {
            /* Enter CAN frame format */
            printf("Enter CAN stream format: (1 - CAN FD stream, 0 - CAN and Notification stream)? : ");
            scanf("%u", &uIsCANFDStreamFormat);

            /* Enter CAN frame data */
            printf("\nEnter CAN frame data:\n");
            printf("Is CAN frame extended (1 - Yes, 0 - No)? : ");
            scanf("%u", &uIsCANFrameExtended);
            printf("Enter CAN ID : ");
            scanf("%lu", &ulCANId);
            /* Checking CAN ID range*/
            if (uIsCANFrameExtended == 0)
            {
                /* Standard CAN ID */
                if (ulCANId > STANDARD_CAN_ID_MAX)
                {
                    printf("Standard CAN ID should not exceed %u\n", STANDARD_CAN_ID_MAX);
                    continue;
                }
            }
            else
            {
                /* Extended CAN ID */
                if (ulCANId > EXTENDED_CAN_ID_MAX)
                {
                    printf("Extended CAN ID should not exceed %lu\n", (long unsigned int)EXTENDED_CAN_ID_MAX);
                    continue;
                }
            }
            uIsCANFDFrame = uIsCANFD_BRS = 0;
            if (uIsCANFDStreamFormat)
            {
                printf("Is CAN frame FD (1 - Yes, 0 - No)? : ");
                scanf("%u", &uIsCANFDFrame);
                if (uIsCANFDFrame > 0)
                {
                    printf("Is BRS required (1 - Yes, 0 - No)? : ");
                    scanf("%u", &uIsCANFD_BRS);
                }
            }
            printf("Enter CAN data length : ");
            scanf("%u", &uCANDataLength);
            /* Checking CAN Data Length*/
            uCANDataLengthMax = uIsCANFDStreamFormat == 1 ? CP_CAN_FD_MAX_DATA_LENGTH : CP_CAN_MAX_DATA_LENGTH;
            if (uCANDataLength > uCANDataLengthMax)
            {
                printf("CAN Data Length should not exceed %u\n", uCANDataLengthMax);
                continue;
            }
            uIsCANRemFrameRequest = 0;
            if (uIsCANFDFrame == 0)
            {
                printf("Is CAN frame a remote request for a data frame (1 - Yes, 0 - No)? : ");
                scanf("%u", &uIsCANRemFrameRequest);
            }
            if (uIsCANRemFrameRequest == 0 || uIsCANFDFrame>0)
            {
                for (i = 0; i < (int)uCANDataLength;i++)
                {
                    printf("Enter CAN Data[%d] = ", i);
                    scanf("%u", &CANData[i]);
                    /* Checking CAN Data */
                    if (CANData[i] > CAN_DATA_MAX)
                    {
                        printf("CAN Data should not exceed %u\n", CAN_DATA_MAX);
                        i--;
                        continue;
                    }
                }
            }
            printf("How many messages would you line to send?\n");
            scanf("%d", &iNumberOfMessages);
            break;
        } while (1);

        do
        {
            for (j = 0; j < iNumberOfMessages; j++)
            {
                if (uIsCANFDStreamFormat > 0)
                {
                    /* Preparing CAN FD frame*/
                    memset(&CANFDFrame, 0, sizeof(CANFDFrame));
                    CANFDFrame.dwID = ulCANId;
                    if (uIsCANFrameExtended > 0) CANFDFrame.bFlags |= CP_CAN_FD_FLAG_EXTENDED_ID;
                    if (uIsCANRemFrameRequest > 0) CANFDFrame.bFlags |= CP_CAN_FD_FLAG_REMOTE_FRAME;
                    if (uIsCANFDFrame > 0) CANFDFrame.bFlags |= CP_CAN_FD_FLAG_FD_FRAME;
                    if (uIsCANFD_BRS > 0) CANFDFrame.bFlags |= CP_CAN_FD_FLAG_FD_BRS;
                    CANFDFrame.bLength = uCANDataLength;
                    if (uIsCANRemFrameRequest == 0 || uIsCANFDFrame > 0)
                    {
                        for (i = 0; i < CANFDFrame.bLength; i++)
                        {
                            CANFDFrame.Data[i] = CANData[i];
                        }
                    }

                    /* Preparing an empty CAN FD Stream message */
                    CPPrepareCANFDStream(&PMessage);

                    /* Adding the CAN FD frame to the CAN FD Stream.
                        There should be enough space, since the message is empty.*/
                    CPAddCANFDFrame(&PMessage, &CANFDFrame, &gcCANFrameRoutingData, 0);
                }
                else
                {
                    /* Preparing CAN frame*/
                    memset(&CANFrame, 0, sizeof(CANFrame));
                    CANFrame.dwId = ulCANId;
                    CANFrame.bEIdFlag = uIsCANFrameExtended == 0 ? CT_FALSE : CT_TRUE;
                    CANFrame.bLength = uCANDataLength;
                    CANFrame.bRemFrameFlag = uIsCANRemFrameRequest == 0 ? CT_FALSE : CT_TRUE;
                    if (!CANFrame.bRemFrameFlag)
                    {
                        for (i = 0; i < (int)uCANDataLength; i++)
                        {
                            CANFrame.Data[i] = CANData[i];
                        }
                    }

                    /* Preparing an empty CAN and Notification Stream message */
                    CPPrepareCANDataAndNotificationStream(&PMessage);

                    /* Adding the CAN frame to the CAN and Notification Stream.
                        There should be enough space, since the message is empty.*/
                    CPAddCANFrame(&PMessage, &CANFrame);
                }

                /* Copying the message to the transmit buffer TxData. */
                bResult = PMCopyToBuffer(&PMessage, TxData, sizeof(TxData), &iBytesToSend);
                assert(bResult);

                /* Sending the CAN and Notification Stream message.*/
                iBytesSent = send(iSocket, TxData, iBytesToSend, 0);
                if (iBytesSent < 0)
                {
                    printf("send() failed with error code: %d\n", GetSocketError());
                    close(iSocket);
                    SocketCleanup();
                    return 1;
                }
                printf("Message #%d. Sent %d bytes.\n", (j+1),iBytesSent);
            }
            printf("\nThe CAN frames have been successfully sent.\n");
            printf("Would you like to repeat (1 - Yes, 0 - No) : ");
            scanf("%u", &uContinue);
        }
        while (uContinue != 0);
        printf("Would you like to continue (1 - Yes, 0 - No) : ");
        scanf("%u", &uContinue);
    } while (uContinue != 0);

    printf("Good bye.\n");
    close(iSocket);
    SocketCleanup();

    return 0;
}
/// @endcond
