/**
******************************************************************************
* @file    Discovery.c
* @project CAN-ENET Software Support Package
*
* @version V1.0.0
* @date    October 2016
* @author  Oleksandr Bogush
*
* @copyright
* (c) Axiomatic Technologies Corp. All rights reserved.
*
* @brief
* This is an example console application showing how the user can discover an
* Axiomatic Ethernet to CAN converter on the local area network (LAN).
*
******************************************************************************
* @attention
*
* <h2><center>&copy; COPYRIGHT(c) 2016 Axiomatic Technologies Corporation</center></h2>
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
*      may be used to endorse or promote products derived from this software
*      without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************
*/
/// @cond
#if defined(_WIN32) || defined(_WIN64) || defined(DOXYGEN)
 #define WINDOWS 1
#else
 #define WINDOWS 0
#endif

#ifndef WINDOWS
 #error Macro WINDOWS should be defined. WINDOWS={0-Not running on Windows, 1-Running on Windows}
#endif

#if WINDOWS==1
  #define _CRT_SECURE_NO_WARNINGS
  #define _WINSOCK_DEPRECATED_NO_WARNINGS

  #include <winsock2.h>
  #include <ws2tcpip.h>
  #pragma comment(lib, "Ws2_32.lib")

  #define SocketCleanup() WSACleanup()
  #define GetSocketError() WSAGetLastError()
  #define close closesocket

  #define MS_PER_TICK (1000.0 / CLOCKS_PER_SEC)
#else
  #include <netinet/in.h>
  #include <netinet/tcp.h>
  #include <sys/socket.h>
  #include <arpa/inet.h>
  #include <errno.h>
  #include <unistd.h>

  #define SocketCleanup()
  #define GetSocketError() errno

  typedef int BOOL;
  #define FALSE 0
  #define TRUE  1

  static double GetTimeDiffInMS(const struct timespec *pTimeStop,const struct timespec *pTimeStart);
#endif

#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <time.h>

#include "../Inc/DiscProtocol.h"

#define SOCKET_TIMEOUT 100 /* [ms] */
#define SOCKET_RECEIVING_TIME 2 /* [s] */

#define HEADER_FORMAT "%17s %15s %10s %10s %14s %12s %12s"

#define MAC_FORMAT "%02X:%02X:%02X:%02X:%02X:%02X "
#define IP_FORMAT "%15s "
#define WEB_PORT_FORMAT "%10u "
#define DEVICE_PORT_FORMAT "%10u "
#define DEVICE_PORT_TYPE_FORMAT "%14s "
#define PART_NUMBER_FORMAT "%12s "
#define SERIAL_NUMBER_FORMAT "%12s"

static void OnDataParsed(PM_PROTOCOL_MESSAGE_t *pPMessage, void *arg);

typedef enum
{
    UDP = 0,
    TCP
}
PORT_TYPE;

int main( int argc, const char* argv[] )
{
    int iResult;
    int iSocket;
    struct sockaddr_in SocketAddr;
    BYTE_t RxData[PM_PROTOCOL_MESSAGE_BUFFER_SIZE];
    PM_PROTOCOL_PARSER_t PParser;
    int iBytesReceived;
    BYTE_t TxData[PM_PROTOCOL_MESSAGE_BUFFER_SIZE];
    PM_PROTOCOL_MESSAGE_t PMessage;
    BOOL_t bResult;
    int iBytesToSend;
    int iBytesSent;
    unsigned int uContinue;
    unsigned int uRequest;
    BOOL bIsDiscoveryResponseReceived;

    #if WINDOWS==1
      WSADATA wsaData;
      BOOL bOptionValue;
      DWORD dwOptionValue;
      clock_t ClockStartTime;
    #else
      int iOptionValue;
      struct timeval TimeoutOptionValue;
      struct timespec ClockStartTime,ClockTime;
    #endif

    printf ("===============================================================\n"
            " Program: Discovery V1.0.0\n"
            " CAN-ENET Software Support Package. p/n AX140910\n"
            " (c) Axiomatic Technologies Corporation\n\n"
            " This is an example program showing how the user can discover an\n"
            " Axiomatic Ethernet to CAN converter on the LAN.\n"
            "===============================================================\n");

    #if WINDOWS==1
      /* We need to initialize Winsock if running on Windows*/
      iResult = WSAStartup(MAKEWORD(2, 2), &wsaData);
      if (iResult != 0)
      {
          printf("WSAStartup() failed with error: %d\n", iResult);
          return 1;
      }
    #endif


    /* First we should create a socket for UDP communication */
    iSocket = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
    if (iSocket <0)
    {
        printf("socket() failed with error code: %d\n", GetSocketError());
        SocketCleanup();
        return 1;
    }

    /* The socket will be used for broadcasting. We are setting the necessary option */
    #if  WINDOWS==1
      bOptionValue = TRUE;
      iResult = setsockopt(iSocket, SOL_SOCKET, SO_BROADCAST, (char*)&bOptionValue, sizeof(bOptionValue));
    #else
      iOptionValue = 1;
      iResult = setsockopt(iSocket, SOL_SOCKET, SO_BROADCAST, &iOptionValue, sizeof(iOptionValue));
    #endif
    if(iResult<0)
    {
        printf("setsockopt() failed with error code: %d\n", GetSocketError());
        close(iSocket);
        SocketCleanup();
        return 1;
    }

    /* Set socket receiving operation timeout. This timeout is necessary in case the converter does not reply to
       the discovery request. */
    #if  WINDOWS==1
      dwOptionValue = SOCKET_TIMEOUT;
      iResult = setsockopt(iSocket, SOL_SOCKET, SO_RCVTIMEO, (char*)&dwOptionValue, sizeof(dwOptionValue));
    #else
      memset(&TimeoutOptionValue, 0, sizeof(TimeoutOptionValue));
      TimeoutOptionValue.tv_sec = SOCKET_TIMEOUT / 1000;
      TimeoutOptionValue.tv_usec = (SOCKET_TIMEOUT - TimeoutOptionValue.tv_sec * 1000) * 1000;
      iResult = setsockopt(iSocket, SOL_SOCKET, SO_RCVTIMEO, &TimeoutOptionValue, sizeof(TimeoutOptionValue));
    #endif
    if (iResult < 0)
    {
        printf("setsockopt() failed with error code: %d\n", GetSocketError());
        close(iSocket);
        SocketCleanup();
        return 1;
    }

    /* Preparing the address data structure for the discovery request. The request is sent to the global IP address,
       DP_DISCOVERY_PORT port. */
    memset(&SocketAddr, 0, sizeof(SocketAddr));
    SocketAddr.sin_family = AF_INET;
    SocketAddr.sin_addr.s_addr = inet_addr("255.255.255.255");
    SocketAddr.sin_port = htons(DP_DISCOVERY_PORT);

    do
    {
        printf("Would you like to send a Discovery Request (1 - Yes, 0 - No) : ");
        scanf("%u", &uRequest);
        if(uRequest!=0)
        {
            /* Preparing a Discovery Request message */
            DPGenRequestMessage(&PMessage);

            /* Copying the message to the transmit buffer TxData. */
            bResult = PMCopyToBuffer(&PMessage, TxData, sizeof(TxData), &iBytesToSend);
            assert(bResult);

            /* Sending the Discovery Request message.*/
            iBytesSent=sendto(iSocket, TxData, iBytesToSend,0,(struct sockaddr *) &SocketAddr, sizeof(SocketAddr));
            if (iBytesSent < 0)
            {
                printf("send() failed with error code: %d\n", GetSocketError());
                close(iSocket);
                SocketCleanup();
                return 1;
            }
            printf("\nThe Discovery Request message has been successfully sent. Waiting for the response...\n");

            /* Initializing the parser */
            memset(&PParser, 0, sizeof(PParser));

            /* Initializing the timing variables. Since POSIX requires that the CLOCKS_PER_SEC value
            be one million independent of the actual resolution, the clock() function cannot be used
            to accurately measure time intervals in Linux. We will use clock_gettime() POSIX function
            instead. */
            #if WINDOWS==1
              ClockStartTime = clock();
            #else
              clock_gettime(CLOCK_MONOTONIC,&ClockStartTime);
            #endif

            /* Initializing the Discovery Response Received flag*/
            bIsDiscoveryResponseReceived = FALSE;

            /* Receiving data in the RxData buffer. */
            do
            {
                iBytesReceived = recv(iSocket, RxData, sizeof(RxData), 0);
                if (iBytesReceived > 0)
                {
                    /* Data has been successfully received.
                    Now we are calling the protocol message parser. */
                    PMParseFromBuffer(RxData, iBytesReceived, &PParser, OnDataParsed, NULL, &bIsDiscoveryResponseReceived);
                }
                else if (iBytesReceived == 0)
                {
                    /* Connection has been closed on the server side */
                    printf("Connection has been closed on the server side.\n");
                    close(iSocket);
                    SocketCleanup();
                    return 1;
                }
                else
                {
                    iResult = GetSocketError();
                    #if  WINDOWS==1
                      if (iResult != WSAETIMEDOUT)
                    #else
                      if (iResult != EAGAIN && iResult != EWOULDBLOCK && iResult != EINPROGRESS)
                    #endif
                    {
                        /* Receiving error */
                        printf("recv() failed with error: %d\n", iResult);
                        close(iSocket);
                        SocketCleanup();
                        return 1;
                    }
                }
            #if WINDOWS==1
            } while (((clock() - ClockStartTime)*MS_PER_TICK) < (SOCKET_RECEIVING_TIME * 1000));
            #else
            clock_gettime(CLOCK_MONOTONIC,&ClockTime);
            } while(GetTimeDiffInMS(&ClockTime,&ClockStartTime)/1000< SOCKET_RECEIVING_TIME);
            #endif

            if (!bIsDiscoveryResponseReceived)
            {
                printf("The %g second(s) waiting time has expired. No Discovery Response message has been received.\n", (double)SOCKET_RECEIVING_TIME);
            }
        }

        printf("Would you like to continue (1 - Yes, 0 - No) : ");
        scanf("%u", &uContinue);
    } while (uContinue != 0);

    printf("Good bye.\n");
    close(iSocket);
    SocketCleanup();

    return 0;
}

/**
  * @brief  This callback function is called on successful parsing of the protocol message.
  *
  * @param[in]  pPMessage: pointer to the parsed protocol message.
  * @param[in]  arg: pointer to an arbitrary data passed to PMParseFromBuffer() function.
  *
  * @retval None
  */
void OnDataParsed(PM_PROTOCOL_MESSAGE_t *pPMessage, void *arg)
{
    DP_DISCOVERY_DATA_t DiscData;
    BOOL *pbIsDiscoveryResponseReceived = (BOOL*)arg;
    char szIpAddr[20];
    BYTE_t *pMACAddr = (BYTE_t*)DiscData.MACAddr.Addr;
    BYTE_t *pIpAddr = (BYTE_t*)DiscData.IpAddr.Addr;

    /* Parsing Communication Protocol Messages. */

    /* Parsing the Discovery Response message. */
    if (DPParseResponse(pPMessage, &DiscData))
    {
        /* Discovery Response message has been parsed */
        if (!*pbIsDiscoveryResponseReceived)
        {
            printf("\n" HEADER_FORMAT "\n\n", "MAC", "IP", "WebPort", "DevPort", "DevPortType", "P/N", "S/N");
        }
        sprintf(szIpAddr, "%hu.%hu.%hu.%hu", pIpAddr[0], pIpAddr[1], pIpAddr[2], pIpAddr[3]);
        printf(MAC_FORMAT IP_FORMAT WEB_PORT_FORMAT DEVICE_PORT_FORMAT DEVICE_PORT_TYPE_FORMAT PART_NUMBER_FORMAT SERIAL_NUMBER_FORMAT"\n",
            pMACAddr[0], pMACAddr[1], pMACAddr[2], pMACAddr[3], pMACAddr[4], pMACAddr[5],
            szIpAddr,DiscData.wWebPort, DiscData.wDevicePort, DPGetPortTypeName(DiscData.DevicePortType), DiscData.szPartNumber, DiscData.szSerialNumber);

        *pbIsDiscoveryResponseReceived = TRUE;
    }
}


#if WINDOWS==0
/**
  * @brief  Calculates time difference in milliseconds between two time stamps
  *         presented by structure timespec in POSIX OS.
  *
  * @param[in]  pTimeStop: pointer to the time stop time stamp.
  * @param[in]  pTimeStart: pointer to the time start time stamp.
  *
  * @retval Time in milliseconds between time start and time stop time stamps.
  */
double GetTimeDiffInMS(const struct timespec *pTimeStop,const struct timespec *pTimeStart)
{
    long lTimeDiffInNS;
    double dTimeDiffInS;

    lTimeDiffInNS=pTimeStop->tv_nsec-pTimeStart->tv_nsec;
    dTimeDiffInS=difftime(pTimeStop->tv_sec,pTimeStart->tv_sec);
    if(lTimeDiffInNS<0)
    {
        lTimeDiffInNS+=1000000000L;
        dTimeDiffInS-=1;
    }
    return dTimeDiffInS*1000.0+((double)lTimeDiffInNS)/1000000.0;
}
#endif
/// @endcond
