/**
  ******************************************************************************
  * @file    CommProtocol.h
  * @project CAN-ENET Software Support Package
  *
  * @version V4.0.0
  * @date    April 2025
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * Header file of the _CommProtocol_ module
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2025 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef CommProtocol_H
#define CommProtocol_H

#ifdef __cplusplus
extern "C" {
#endif

/* Includes ------------------------------------------------------------------*/
#include "PMessage.h"

/** @defgroup CAN_ENET_SSP CAN-ENET Software Support Package
  * @{
  */

/** @defgroup CommProtocol Communication Protocol
  * Communication Protocol Module
  * @{
  */
/* Exported constants --------------------------------------------------------*/
/** @defgroup CommProtocol_Exported_Constants Exported Constants
  * @{
  */

/** @defgroup CommProtocol_CAN_Max_DataLength CAN and CAN FD Maximum Data Length
  * @{
  */
#define CP_CAN_MAX_DATA_LENGTH          8
#define CP_CAN_FD_MAX_DATA_LENGTH       64
/**
  * @}
  */

/** @defgroup CommProtocol_CAN_FD_Frame_Flags CAN FD Frame Flags
  * @{
  */
#define CP_CAN_FD_FLAG_ERROR_MESSAGE    0x80
#define CP_CAN_FD_FLAG_EXTENDED_ID      0x40
#define CP_CAN_FD_FLAG_REMOTE_FRAME     0x20
#define CP_CAN_FD_FLAG_FD_FRAME         0x10
#define CP_CAN_FD_FLAG_FD_BRS           0x08
#define CP_CAN_FD_FLAG_FD_ESI           0x04
/**
  * @}
  */
  
/** @defgroup CommProtocol_CAN_FD_Error_Codes CAN FD Error Codes
  * @{
  */

/**
  * @brief CAN FD Error Codes. Set in Data[0] when CP_CAN_FD_FLAG_ERROR_MESSAGE flag is set
  */
#define CP_CAN_ERROR_UNDEFINED          0
#define CP_CAN_ERROR_CAN_WARNING        0x01
#define CP_CAN_ERROR_CAN_PASSIVE        0x02
#define CP_CAN_ERROR_CAN_BUS_OFF        0x03
/**
  * @}
  */

/** @defgroup CommProtocol_SupportedFeatures Supported Feature Flags for Communication Node
  * @{
  */
#define CP_SUPPORTED_FEATURE_FLAG_CAN_FD_STREAM_ONE_FRAME_PER_MESSAGE   ((1U)<<1)
#define CP_SUPPORTED_FEATURE_FLAG_CAN_FD_STREAM                         ((1U)<<0)

#define MAX_NUMBER_OF_COMM_NODE_FILTERS 10
/**
  * @}
  */

/**
  * @defgroup CommProtocol_CANAndNotificationStreamAddress CAN Address (ChannelGroup,ChannelIDSet) of the CAN and Notification Stream
  * @{
  */
#define CP_CAN_AND_NOTIFICATION_STREAM_GROUP              0
#define CP_CAN_AND_NOTIFICATION_STREAM_CHANNEL_ID_SET     (1U<<0)

#define CP_CAN_AND_NOTIFICATION_STREAM_CHANNEL            1
/**
  * @}
  */

/**
  * @brief Heartbeat Message interval in ms
  */
#define CP_HEARTBEAT_MESSAGE_INTERVAL 1000

/**
  * @brief Timeout on inactivity in ms
  *
  * If the UDP protocol is used, the connection is considered to be lost after
  * a period of inactivity on one of the nodes. Inactivity means no protocol
  * messages for a certain period of time.
  */
#define CP_INACTIVITY_TIMEOUT 10000

/**
  * @}
  */
/* Exported macros -----------------------------------------------------------*/
/**
  * @defgroup CommProtocol_Exported_Macros Exported Macros
  * @{
  */

/**
  * @brief Macro converting CAN Channel Number into CAN Channel ID
  *
  * @note CAN Channel Number range is: 1...32
  *       CAN Channel IDs generated by this macro can be ORed
  */
#define CP_CalcChannelID(CANChannelNumber) (((CANChannelNumber) < 1 || (CANChannelNumber)>32) ? 0 : (1UL << ((CANChannelNumber) - 1)))

/**
  * @brief Macro Checking if CAN Channel ID is included in the CAN Channel ID Set
  *
  */
#define CP_IsChannelIDIncluded(ChannelIDSet,ChannelID) (((ChannelIDSet) & (ChannelID))>0)

/**
  * @brief Macro Setting CAN Address
  *
  */
#define CP_SetCANAddr(CANAddr,ChannelGroup,ChannelIDSet) {(CANAddr).bChannelGroup=(ChannelGroup); (CANAddr).dwChannelIDSet=(ChannelIDSet);}

/**
  * @brief Macro checking if two CAN Addresses match
  *
  */
#define CP_IsCANAddrMatch(CANAddr1,CANAddr2) (((CANAddr1).bChannelGroup==(CANAddr2).bChannelGroup) && CP_IsChannelIDIncluded((CANAddr1).dwChannelIDSet,(CANAddr2).dwChannelIDSet))

/**
  * @brief Macro checking if two CAN Addresses are equal
  *
  */
#define CP_IsCANAddrEqual(CANAddr1,CANAddr2) (((CANAddr1).bChannelGroup==(CANAddr2).bChannelGroup) && ((CANAddr1).dwChannelIDSet==(CANAddr2).dwChannelIDSet))

/**
  * @brief Macro checking if CAN Addresses is valid
  *
  */
#define CP_IsCANAddrValid(CANAddr) ((CANAddr).dwChannelIDSet!=0)

/**
  * @brief Macro checking if CAN Addresses is NULL
  *
  */
#define CP_IsCANAddrNULL(CANAddr) ((((CANAddr).bChannelGroup)==0) && (((CANAddr).dwChannelIDSet)==0))

/**
  * @brief Macro checking if CAN Address matches the CAN and Notification Stream address
  *
  */
#define CP_IsCANAndNotificationFrameStreamAddrMatch(CANAddr) ((((CANAddr).bChannelGroup)==CP_CAN_AND_NOTIFICATION_STREAM_GROUP) && CP_IsChannelIDIncluded((CANAddr).dwChannelIDSet,CP_CAN_AND_NOTIFICATION_STREAM_CHANNEL_ID_SET))

/**
  * @}
  */
/* Exported types ------------------------------------------------------------*/

/** @defgroup CommProtocol_Exported_Types Exported Types
  * @{
  */

/**
  * @brief Converter Type
  */
typedef enum
{
    CP_CONVERTER_CAN_ETHERNET=0,   /**< Ethernet to CAN converter with CAN Voltage Output */
    CP_CONVERTER_CAN_WIFI,         /**< WiFi to CAN converter */
    CP_CONVERTER_CAN_WIFI_DATALOG, /**< WiFi to CAN converter with CAN datalogging capability */
    CP_CONVERTER_GENERIC           /**< Generic Converter */
}
CP_CONVERTER_TYPE_t;

/**
  * @brief CAN Frame. Deprecated in SSP v3.0.0
  */
typedef struct
{
    DWORD_t dwId;                         /**< CAN Identifier (CAN ID)*/
    BOOL_t bEIdFlag;                      /**< Extended CAN ID Flag {0 - Standard CAN ID, 1 - Extended CAN ID} */
    BOOL_t bRemFrameFlag;                 /**< Remote Frame Flag {0 - Regular data frame, 1 - Remote request for a data frame} */
    BYTE_t bLength;                       /**< Data Length (up to 8 bytes)*/
    BYTE_t Data[CP_CAN_MAX_DATA_LENGTH];  /**< CAN Data Array */

    DWORD_t dwTimeStamp;  /**< Optional Time Stamp. Carries a time interval between the current
                            *  and the previous CAN messages received on the CAN bus.
                            */
}
CP_CAN_FRAME_t;

/**
  * @brief Notification Frame. Deprecated in SSP v3.0.0
  */
typedef struct
{
    BYTE_t bId;      /**< Notification Identifier (0...127)*/
    DWORD_t dwData;  /**< Notification Data */
}
CP_NOTIFICATION_FRAME_t;

/**
  * @brief CAN FD Frame. Can be used for both: Classical CAN and CAN FD frames
  */
typedef struct
{
    DWORD_t dwID;                            /**< CAN Identifier (CAN ID)*/
    BYTE_t bFlags;                           /**< CAN FD Frame Flags */
    BYTE_t bLength;                          /**< Data Length (up to 64 bytes)*/
    BYTE_t Data[CP_CAN_FD_MAX_DATA_LENGTH];  /**< CAN Data Array */
}
CP_CAN_FD_FRAME_t;

/**
  * @brief CAN Address
  */
typedef struct
{
    BYTE_t  bChannelGroup;        /**< CAN Channel Group. First part of the CAN Address  (bChannelGroup,dwChannelIDSet).
                                    *  CAN Channel Group can be defined from 0 to 255. */
    DWORD_t dwChannelIDSet;       /**< CAN Channel ID Set. Second part of the CAN Address  (bChannelGroup,dwChannelIDSet).
                                    *  Each CAN channel in the CAN Channel ID Set is presented by a bit (flag) at the corresponding CAN channel number position.
                                    *  If the bit is set, the CAN Channel is included in this address. Up to 32 CAN Channels, or ports, can be simultaneously addressed in a given CAN Channel Group.
                                    *  If dwChannelIDSet=0, the CAN Address is invalid. It does not address any CAN Channels. 
                                    *  If (bChannelGroup,dwChannelIDSet)=(0,0) - the CAN Address is NULL. This form of invalid CAN Address is used in CAN Address filtering.
                                    *  Legacy devices supporting old versions of the Communication Protocol are assumed to use 
                                    *  CAN and Notification Stream Address = (0,1)
                                    */
}
CP_CAN_ADDRESS_t;

/**
  * @brief CAN FD Frame routing data
  *
  * @note  CAN FD Frames are routed based on the value of the CAN Frame Address: CANAddr. The structure also contains Physical Channel Number of the CAN port on the device: wPhysicalChannelNumber
  *        and 3 flags: wFlags (only 3 most significant bits are valid: brrrxxxxxxxxxxxxx, r={0,1} - flag value). The Flags can be used in the future extensions of the protocol.
  */
typedef struct
{
    WORD_t wFlags;                  /**< Flags reserved for the future use. Only 3 most significant bits are valid. Should be set to zero. */
    WORD_t wPhysicalChannelNumber;  /**< CAN port Physical Channel Number in the controller (0...8191). 0 - if undefined */

    CP_CAN_ADDRESS_t CANAddr;      /**< CAN Frame Address. The transmitting CAN port places here its Transmitting CAN Port Address.
                                     *  A receiving CAN port receives the CAN FD Frame if this address matches the Receiving CAN Port Address of the receiving CAN Port.
                                     */
}
CP_CAN_FRAME_ROUTING_DATA_t;

/**
  * @brief Communication Node Filter List. If a CAN Frame Address matches one of the filters in the list, the frame passes the filter and is sent to the node. If the filter is disabled 
  *       (bNumberOfFilters=0), all frames with valid CAN Frame Addresses will pass the filter and will be sent to the node.
  */
typedef struct
{
    CP_CAN_ADDRESS_t Filter[MAX_NUMBER_OF_COMM_NODE_FILTERS]; /**< Array of communication node filters (CAN Addresses) used for filtering CAN FD Frames transmitted to the node */
    BYTE_t bNumberOfFilters;                                  /**< Number of communication node filters (CAN Addresses) in the array (0...MAX_NUMBER_OF_COMM_NODE_FILTERS). 0 - filter is disabled  */
}
CP_COMMUNICATION_NODE_FILTER_LIST_t;

/** @defgroup CommProtocol_Callback_Functions Callback Function Prototypes
  * @{
  */

/** @defgroup CommProtocol_Callback_Functions_CPParseCANDataAndNotificationStream CPParseCANDataAndNotificationStream() Callback Function Prototypes. Deprecated in SSP v3.0.0
  * @{
  */
typedef void CP_ON_CAN_FRAME_PARSED_FUNCTION_t(CP_CAN_FRAME_t *pCANFrame,void *arg);
typedef void CP_ON_NOTIFICATION_FRAME_PARSED_FUNCTION_t(CP_NOTIFICATION_FRAME_t *pNotificationFrame,void *arg);
/**
  * @}
  */
  
/** @defgroup CommProtocol_Callback_Functions_CPParseCANFDStream CPParseCANFDStream() Callback Function Prototype
  * @{
  */
typedef void CP_ON_CAN_FD_FRAME_PARSED_FUNCTION_t(CP_CAN_FD_FRAME_t *pCANFDFrame,CP_CAN_FRAME_ROUTING_DATA_t *pCANFrameRoutingData,DWORD_t dwAbsTimeStamp,void *arg);
/**
  * @}
  */
  
/**
  * @}
  */
  
/**
  * @}
  */
/* Exported functions --------------------------------------------------------*/
/** @defgroup CommProtocol_Exported_Functions Exported Functions
  * @{
  */

/** @defgroup CommProtocol_Exported_Deprecated_Functions CAN Data and Notification stream functions. Deprecated in SSP v3.0.0
  * @{
  */
BOOL_t CPParseCANDataAndNotificationStream(const PM_PROTOCOL_MESSAGE_t *pPMessage,CP_ON_CAN_FRAME_PARSED_FUNCTION_t OnCANFrameParsed,CP_ON_NOTIFICATION_FRAME_PARSED_FUNCTION_t OnCANNotificationFrameParsed,void *arg);
void CPPrepareCANDataAndNotificationStream(PM_PROTOCOL_MESSAGE_t *pPMessage);
BOOL_t CPAddCANFrame(PM_PROTOCOL_MESSAGE_t *pPMessage,const CP_CAN_FRAME_t *pCANFrame);
BOOL_t CPAddNotificationFrame(PM_PROTOCOL_MESSAGE_t *pPMessage,const CP_NOTIFICATION_FRAME_t *pNotificationFrame);
BOOL_t CPIsCANFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage);
BOOL_t CPIsNotificationFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage);
/**
  * @}
  */

/** @defgroup CommProtocol_Exported_CAN_FD_Stream_Functions CAN FD stream functions
  * @{
  */
BOOL_t CPParseCANFDStream(const PM_PROTOCOL_MESSAGE_t *pPMessage,CP_ON_CAN_FD_FRAME_PARSED_FUNCTION_t OnCANFDFrameParsed,void *arg);
void CPPrepareCANFDStream(PM_PROTOCOL_MESSAGE_t *pPMessage);
BOOL_t CPAddCANFDFrame(PM_PROTOCOL_MESSAGE_t *pPMessage,const CP_CAN_FD_FRAME_t *pCANFDFrame,const CP_CAN_FRAME_ROUTING_DATA_t *pCANFrameRoutingData,DWORD_t dwAbsoluteTimeStamp);
BOOL_t CPIsCANFDFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage);
BOOL_t CPIsCANClassicalCANFDFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage);
/**
  * @}
  */

BOOL_t CPParseStatusRequest(const PM_PROTOCOL_MESSAGE_t *pPMessage);
BOOL_t CPParseStatusResponse(const PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t *pdwHealthData,DWORD_t *pdwCANRxDErrors,DWORD_t *pdwCANTxDErrors,DWORD_t *pdwCANBusOffErrors,CP_CONVERTER_TYPE_t *pConverterType,
                             DWORD_t *pCommNodeSupportedFeatures,CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);
BOOL_t CPParseHeartbeat(const PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t *pdwMessageNumber,DWORD_t *pdwTimeInterval,DWORD_t *pdwHealthData,CP_CONVERTER_TYPE_t *pConverterType,
	                           DWORD_t *pCommNodeSupportedFeatures,CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);

void CPGenStatusRequestMessage(PM_PROTOCOL_MESSAGE_t *pPMessage);
void CPGenStatusResponseMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t dwHealthData,DWORD_t dwCANRxDErrors,DWORD_t dwCANTxDErrors,DWORD_t dwCANBusOffErrors,CP_CONVERTER_TYPE_t ConverterType, 
                                DWORD_t dwCommNodeSupportedFeatures,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);
void CPGenHearbeatMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t dwMessageNumber,DWORD_t dwTimeInterval,DWORD_t dwHealthData,CP_CONVERTER_TYPE_t ConverterType,
                        DWORD_t dwCommNodeSupportedFeatures,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);

BOOL_t CPIsCANAddressPassCommNodeFilter(const CP_CAN_ADDRESS_t *pCANAddr,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);

/** @defgroup CommProtocol_Exported_Converting_Functions Functions converting between CAN and CAN FD frame formats
  * @{
  */
void CPConvertCANToCANFDFrame(CP_CAN_FD_FRAME_t *pCANFDFrame,const CP_CAN_FRAME_t *pCANFrame);
BOOL_t CPConvertCANFDToCANrame(CP_CAN_FRAME_t *pCANFrame,const CP_CAN_FD_FRAME_t *pCANFDFrame,DWORD_t dwRelativeTimeStamp);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* CommProtocol_H */
