/**
  ******************************************************************************
  * @file    DiscProtocol.h
  * @project CAN-ENET Software Support Package
  *
  * @version V4.0.1
  * @date    December 2025
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * Header file of the _DiscProtocol_ module
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2025 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef DiscProtocol_H
#define DiscProtocol_H

#ifdef __cplusplus
extern "C" {
#endif
/* Includes ------------------------------------------------------------------*/
#include "PMessage.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @defgroup DiscProtocol Discovery Protocol
  * Discovery Protocol Module
  * @{
  */

/* Exported constants --------------------------------------------------------*/
/** @defgroup DiscProtocol_Exported_Constants Exported Constants
  * @{
  */
#define DP_DISCOVERY_PORT 35100 /**< Communication port used for discovery requests. */

/** @defgroup DiscProtocol_String_Field_Limit String Field Limitations
  * Limitations for string fields in DP_DISCOVERY_DATA structure. The terminating '\0' is not counted.
  * @{
  */
#define DP_PART_NUMBER_STRING_LENGTH_MAX   25
#define DP_SERIAL_NUMBER_STRING_LENGTH_MAX 15
/**
  * @}
  */
/**
  * @}
  */
/* Exported types ------------------------------------------------------------*/
/** @defgroup DiscProtocol_Exported_Types Exported Types
  * @{
  */

/**
  * @brief MAC Address: MAC_ADDR0:MAC_ADDR1:MAC_ADDR2:MAC_ADDR3:MAC_ADDR4:MAC_ADDR5
  */
typedef struct
{
    /* MAC ADDRESS: MAC_ADDR0:MAC_ADDR1:MAC_ADDR2:MAC_ADDR3:MAC_ADDR4:MAC_ADDR5 */
    BYTE_t Addr[6];
}
DP_MAC_ADDR_t;

/**
  * @brief IP Address: Addr0.Addr1.Addr2.Addr3
  */
typedef struct
{
    // IP Address: Addr0.Addr1.Addr2.Addr3
    BYTE_t Addr[4];
}
DP_IP_ADDR_t;

/**
  * @brief Port Type
  */
typedef enum
{
    DP_PORT_TYPE_UDP=0, /**< UDP Port */
    DP_PORT_TYPE_TCP    /**< TCP Port */
}
DP_PORT_TYPE_t;

/**
  * @brief Discovery Data
  */
typedef struct
{
  DP_MAC_ADDR_t MACAddr;
  DP_IP_ADDR_t IpAddr;
  WORD_t wWebPort;
  WORD_t wDevicePort;
  DP_PORT_TYPE_t DevicePortType;
  char szPartNumber[DP_PART_NUMBER_STRING_LENGTH_MAX+1];
  char szSerialNumber[DP_SERIAL_NUMBER_STRING_LENGTH_MAX+1];
}
DP_DISCOVERY_DATA_t;
/**
  * @}
  */
/* Exported functions --------------------------------------------------------*/
/** @defgroup DiscProtocol_Exported_Functions Exported Functions
  * @{
  */
BOOL_t DPParseRequest(const PM_PROTOCOL_MESSAGE_t *pPMessage);
void DPGenResponseMessage(PM_PROTOCOL_MESSAGE_t *pPMessage, const DP_DISCOVERY_DATA_t *pDiscData);
void DPGenRequestMessage(PM_PROTOCOL_MESSAGE_t *pPMessage);
BOOL_t DPParseResponse(const PM_PROTOCOL_MESSAGE_t *pPMessage,DP_DISCOVERY_DATA_t *pDiscData);
BOOL_t DPIsDiscDataEqual(const DP_DISCOVERY_DATA_t *pDiscData1,const DP_DISCOVERY_DATA_t *pDiscData2);
const char *DPGetPortTypeName(DP_PORT_TYPE_t PortType);
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* DiscProtocol_H */
/**
  * @}
  */

/**
  * @}
  */

