/**
  ******************************************************************************
  * @file    HealthData.h
  * @project CAN-ENET Software Support Package
  *
  * @version V4.0.1
  * @date    April 2025
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * Header file of the _HealthData_ module
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2025 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef HealthData_H
#define HealthData_H

#ifdef __cplusplus
extern "C" {
#endif
/* Includes ------------------------------------------------------------------*/
#include "CommProtocol.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @defgroup HealthData Health Data
  * Health Data module
  * @{
  */
/* Exported definitions ------------------------------------------------------*/
/* Exported constants --------------------------------------------------------*/
/* Exported types ------------------------------------------------------------*/
/** @defgroup HealthData_Exported_Types Exported Types
  * @{
  */
/**
  * @brief Operational Statuses
  */
typedef enum
{
    HD_OPERATIONAL_STATUS_UNDEF=0,  /**< Undefined */
    HD_OPERATIONAL_STATUS_NORMAL,   /**< Normal */
    HD_OPERATIONAL_STATUS_WARNING,  /**< Warning */
    HD_OPERATIONAL_STATUS_ERROR     /**< Error */
}
HD_OPERATIONAL_STATUS_t;

/**
  * @brief Converter Components
  */
typedef enum
{
    HD_CONVERTER_CONTROLLER_CP_LOAD=0, /**< The result of the loading operation for common configuration parameters. */
    HD_CONVERTER_CAN_CP_LOAD,          /**< The result of the loading operation for CAN configuration parameters. */
    HD_CONVERTER_ETH_CP_LOAD,          /**< The result of the loading operation for Ethernet configuration parameters. Only for Ethernet-CAN converters. */
    HD_CONVERTER_WIFI_CP_LOAD=HD_CONVERTER_ETH_CP_LOAD,         /**< The result of the loading operation for WiFi configuration parameters. Only for WiFi-CAN converters. */

    HD_CONVERTER_CAN_DRIVER_INIT,      /**< The result of the CAN driver initialization. */
    HD_CONVERTER_ETH_DRIVER_INIT,      /**< The result of the Ethernet driver initialization. Only for Ethernet-CAN converters. */
    HD_CONVERTER_WIFI_DRIVER_INIT=HD_CONVERTER_ETH_DRIVER_INIT,     /**< The result of the WiFi driver initialization. Only for WiFi-CAN converters. */
    HD_CONVERTER_LWIP_STACK_INIT,      /**< The result of the LwIp stack initialization. */
    HD_CONVERTER_IP_STACK_INIT=HD_CONVERTER_LWIP_STACK_INIT, /**< The result of the IP stack initialization. Generic version */

    HD_CONVERTER_CAN_OPERATION,        /**< Run-time CAN status. */
    HD_CONVERTER_ETH_OPERATION,        /**< Run-time Ethernet status. Only for Ethernet-CAN converters. */
    HD_CONVERTER_WIFI_OPERATION=HD_CONVERTER_ETH_OPERATION,        /**< Run-time WiFi status. Only for WiFi-CAN converters. */

    HD_CONVERTER_POWER_SUPPLY_VOLTAGE,  /**< Power Supply Voltage status. */
    HD_CONVERTER_OUTPUT_CAN_VOLTAGE,    /**< Output CAN Voltage status. Only for Ethernet-CAN converters. */
    HD_CONVERTER_INTERNAL_BAT_VOLTAGE,  /**< Internal microcontroller voltage state. */
    HD_CONVERTER_INTERNAL_uC_VOLTAGE=HD_CONVERTER_INTERNAL_BAT_VOLTAGE, /**< Internal microcontroller voltage state. Generic Version*/
    HD_CONVERTER_UC_TEMPERATURE,        /**< Microcontroller temperature state. */

    HD_CONVERTER_FLASH_MEMORY_DRIVER_INIT, /**< The result of the external flash memory driver initialization. */
    HD_CONVERTER_LOGGING_MEMORY_DRIVER_INIT, /**< The result of the logging memory driver initialization. Only for CAN data-logging converters.*/
    HD_CONVERTER_CAN_LOGGING_OPERATION,      /**< CAN logging operation run-time status. Only for CAN data-logging converters.*/

    HD_NUMBER_OF_CONVERTER_COMPONENTS
}
HD_CONVERTER_COMPONENT_t;

/**
  * @brief Converter Health Data
  */
typedef struct
{
    HD_OPERATIONAL_STATUS_t OperationalStatus[HD_NUMBER_OF_CONVERTER_COMPONENTS];
}
HD_HEALTH_DATA_t;
/**
  * @}
  */
/* Exported functions --------------------------------------------------------*/
/** @defgroup HealthData_Exported_Functions Exported Functions
  * @{
  */
const char *HDGetConverterTypeName(CP_CONVERTER_TYPE_t ConverterType);
const char *HDGetOperationalStatusName(HD_OPERATIONAL_STATUS_t OperationalStatus);
const char *HDGetConverterComponentName(HD_CONVERTER_COMPONENT_t ConverterComponent,CP_CONVERTER_TYPE_t ConverterType);
BOOL_t HDIsConverterComponentInUse(HD_CONVERTER_COMPONENT_t ConverterComponent,CP_CONVERTER_TYPE_t ConverterType);
DWORD_t HDPackHealthData(const HD_HEALTH_DATA_t *pHealthData,CP_CONVERTER_TYPE_t ConverterType);
HD_OPERATIONAL_STATUS_t HDUnpackHealthData(DWORD_t dwHealthData,HD_HEALTH_DATA_t *pHealthData,CP_CONVERTER_TYPE_t ConverterType);
HD_OPERATIONAL_STATUS_t HDCalcConverterHealthStatus(const HD_HEALTH_DATA_t *pHealthData,CP_CONVERTER_TYPE_t ConverterType);
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* HealthData_H */
/**
  * @}
  */

/**
  * @}
  */
