/**
  ******************************************************************************
  * @file    PMessage.h
  * @project CAN-ENET Software Support Package
  *
  * @version V2.0.0
  * @date    April 2021
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * Header file of the _CommProtocol_ module
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2021 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef PMessage_H
#define PMessage_H

#ifdef __cplusplus
extern "C" {
#endif
/* Includes ------------------------------------------------------------------*/
#include <stddef.h>

#include "CommonTypes.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @defgroup PMessage Protocol Message
  * Protocol Message Module
  * @{
  */


/* Exported constants --------------------------------------------------------*/
/** @defgroup PMessage_Exported_Constants Exported Constants
  * @{
  */

/**
  * @brief Protocol Message buffer size. Includes both: message header and data fields.
  */
#define PM_PROTOCOL_MESSAGE_BUFFER_SIZE     256

/** @defgroup PMessage_Structure_Constants Protocol Message Format Constants
  * @{
  */
// Protocol message structure
#define PM_PROTOCOL_HEADER_INDEX  0
#define PM_PROTOCOL_HEADER_LENGTH 4
#define PM_PROTOCOL_ID_INDEX  (PM_PROTOCOL_HEADER_INDEX + PM_PROTOCOL_HEADER_LENGTH)
#define PM_PROTOCOL_ID_LENGTH 2
#define PM_PROTOCOL_MESSAGE_ID_INDEX  (PM_PROTOCOL_ID_INDEX+PM_PROTOCOL_ID_LENGTH)
#define PM_PROTOCOL_MESSAGE_ID_LENGTH   2
#define PM_PROTOCOL_MESSAGE_VERSION_INDEX   (PM_PROTOCOL_MESSAGE_ID_INDEX+PM_PROTOCOL_MESSAGE_ID_LENGTH)
#define PM_PROTOCOL_MESSAGE_VERSION_LENGTH  1
#define PM_PROTOCOL_MESSAGE_DATA_LENGTH_INDEX (PM_PROTOCOL_MESSAGE_VERSION_INDEX+PM_PROTOCOL_MESSAGE_VERSION_LENGTH)
#define PM_PROTOCOL_MESSAGE_DATA_LENGTH_LENGTH 2
#define PM_PROTOCOL_MESSAGE_DATA_INDEX (PM_PROTOCOL_MESSAGE_DATA_LENGTH_INDEX+PM_PROTOCOL_MESSAGE_DATA_LENGTH_LENGTH)

#define PM_PROTOCOL_MESSAGE_DATA_SIZE (PM_PROTOCOL_MESSAGE_BUFFER_SIZE-PM_PROTOCOL_MESSAGE_DATA_INDEX)
/**
  * @}
  */
/**
  * @}
  */

/* Exported types ------------------------------------------------------------*/
/** @defgroup PMessage_Exported_Types Exported Types
  * @{
  */
/**
  * @brief Protocol Message
  */
typedef struct
{
    WORD_t wProtocolID; /**< Protocol ID from the Message Header */
    WORD_t wMessageID; /**< Message ID from the Message Header */
    BYTE_t bMessageVersion; /**< Message Version from the Message Header */
    WORD_t wDataLength; /**< Data Length from the Message Header */
    BYTE_t Data[PM_PROTOCOL_MESSAGE_DATA_SIZE]; /*!< Message Data */
}
PM_PROTOCOL_MESSAGE_t;

/**
  * @brief Protocol Message parser states
  */
typedef enum
{
    PM_PARSER_STATE_INIT = 0,
    PM_PARSER_STATE_HEADER,
    PM_PARDER_STATE_PROTOCOL_ID,
    PM_PARSER_STATE_MESSAGE_ID,
    PM_PARSER_STATE_MESSAGE_VERSION,
    PM_PARSER_STATE_DATA_LENGTH,
    PM_PARSER_STATE_DATA
}
PM_PARSER_STATE_t;

/**
  * @brief Protocol Message parser
  */
typedef struct
{
    PM_PROTOCOL_MESSAGE_t PMessage;
    PM_PARSER_STATE_t ParserState;
    int iIndex;
    WORD_t wData;
}
PM_PROTOCOL_PARSER_t;

/** @defgroup PMessage_Callback_functions PMParseFromBuffer() Callback Function Prototypes
  * @{
  */
typedef void PM_ON_DATA_PARSED_FUNCTION_t(PM_PROTOCOL_MESSAGE_t *pPMessage,void *arg);
typedef void PM_ON_DATA_PARSED_ERROR_FUNCTION_t(void *arg);
/**
  * @}
  */
/**
  * @}
  */

/* Exported functions --------------------------------------------------------*/
/** @defgroup PMessage_Exported_Functions Exported Functions
  * @{
  */
BOOL_t PMCopyToBuffer(const PM_PROTOCOL_MESSAGE_t *pPMessage,BYTE_t *pBuffer,size_t BufferSize,int *piBytesToSend);
void PMParseFromBuffer(const BYTE_t *pBuffer, int iDataLength, PM_PROTOCOL_PARSER_t *pPParser, PM_ON_DATA_PARSED_FUNCTION_t OnDataParsed, PM_ON_DATA_PARSED_ERROR_FUNCTION_t OnDataParsedError, void *arg);

void PMSetDataToMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,int iIndex,int iSize,DWORD_t dwData);
DWORD_t PMGetDataFromMessage(const PM_PROTOCOL_MESSAGE_t *pPMessage,int iIndex,int iSize);
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* PMessage_H */
/**
  * @}
  */

/**
  * @}
  */
