/**
  ******************************************************************************
  * @file    CommProtocol.c
  * @project CAN-ENET Software Support Package
  *
  * @version V4.0.0
  * @date    April 2025
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * This file provides functions to support messages from the \ref Documentation
  * "Ethernet to CAN Converter Communication Protocol. Version: 6". The protocol
  * independent message structure is supported in \link PMessage.h PMessage\endlink
  * module.
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2025 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include <assert.h>

#include "../Inc/CommProtocol.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @addtogroup CommProtocol
  * @{
  */
/* Private constants ---------------------------------------------------------*/
/** @defgroup CommProtocol_Private_Constants Private Constants
  * @{
  */

/** @defgroup CommProtocol_CAN_Frame_Constants CAN Frame Constants. Deprecated in SSP v3.0.0
  * @{
  */
#define RF_ID_FLAG  0x00008000
#define RF_EID_FLAG 0x80000000
#define ID_MASK     0x000007ff
#define EID_MASK    0x1fffffff

#define FIRST_BYTE_EID_FLAG  0x10
#define CAN_FRAME_LENGTH_MAX 17
/**
  * @}
  */

/** @defgroup CommProtocol_Notification_Frame_Constants Notification Frame Constants. Deprecated in SSP v3.0.0
  * @{
  */
#define FIRST_BYTE_NOTIFICATION_FRAME_FLAG 0x80
#define NOTIFICATION_FRAME_LENGTH 5
/**
  * @}
  */

/** @defgroup CommProtocol_CAN_FD_Frame_Constants CAN FD Frame Constants
  * @{
  */
#define PHYSICAL_CHANNEL_NUMBER_MASK 0x1fff
#define FD_ID_MASK                   0x1fffffff
/**
  * @}
  */

/**
  * @brief Communication Protocol ID
  */
#define COMM_PROTOCOL_ID 14010

/** @defgroup CommProtocol_CAN_Address_Constants CAN Address Constants
  * @{
  */
#define CP_CAN_ADDRESS_LENGTH 5
#define CP_CAN_ADDRESS_CHANNEL_GROUP_LENGTH 1
#define CP_CAN_ADDRESS_CHANNEL_ID_SET_LENGTH 4

/** @defgroup CommProtocol_Heartbeat_Format Heartbeat Message Format Constants
  * @{
  */
//---------------------------------------------------------------------------------------------------
#define HEARTBEAT_MESSAGE_NUMBER_INDEX         0
#define HEARTBEAT_MESSAGE_NUMBER_LENGTH        4
#define HEARTBEAT_TIME_INTERVAL_INDEX          (HEARTBEAT_MESSAGE_NUMBER_INDEX+HEARTBEAT_MESSAGE_NUMBER_LENGTH)
#define HEARTBEAT_TIME_INTERVAL_LENGTH         4
#define HEARTBEAT_HEALTH_DATA_INDEX            (HEARTBEAT_TIME_INTERVAL_INDEX+HEARTBEAT_TIME_INTERVAL_LENGTH)
#define HEARTBEAT_HEALTH_DATA_LENGTH           4
#define HEARTBEAT_CONVERTER_TYPE_INDEX         (HEARTBEAT_HEALTH_DATA_INDEX+HEARTBEAT_HEALTH_DATA_LENGTH)
#define HEARTBEAT_CONVERTER_TYPE_LENGTH        1
#define HEARTBEAT_SUPPORTED_FEATURES_INDEX     (HEARTBEAT_CONVERTER_TYPE_INDEX+HEARTBEAT_CONVERTER_TYPE_LENGTH)
#define HEARTBEAT_SUPPORTED_FEATURES_LENGTH    4
#define HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_INDEX          (HEARTBEAT_SUPPORTED_FEATURES_INDEX+HEARTBEAT_SUPPORTED_FEATURES_LENGTH)
#define HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_LENGTH         1
#define HEARTBEAT_MAIN_FILTER_CHANNEL_ID_SET_INDEX         (HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_INDEX+HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_LENGTH)
#define HEARTBEAT_MAIN_FILTER_CHANNEL_ID_SET_LENGTH        4
#define HEARTBEAT_NUMBER_OF_ADDITIONAL_FILTERS_INDEX       (HEARTBEAT_MAIN_FILTER_CHANNEL_ID_SET_INDEX+HEARTBEAT_MAIN_FILTER_CHANNEL_ID_SET_LENGTH)
#define HEARTBEAT_NUMBER_OF_ADDITIONAL_FILTERS_LENGTH 1
#define HEARTBEAT_MESSAGE_DATA_LENGTH          (HEARTBEAT_MESSAGE_NUMBER_LENGTH+\
                                                HEARTBEAT_TIME_INTERVAL_LENGTH+\
                                                HEARTBEAT_HEALTH_DATA_LENGTH+\
                                                HEARTBEAT_CONVERTER_TYPE_LENGTH+\
                                                HEARTBEAT_SUPPORTED_FEATURES_LENGTH+\
                                                HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_LENGTH+\
                                                HEARTBEAT_MAIN_FILTER_CHANNEL_ID_SET_LENGTH+\
                                                HEARTBEAT_NUMBER_OF_ADDITIONAL_FILTERS_LENGTH)
/**
  * @}
  */
//---------------------------------------------------------------------------------------------------
/** @defgroup CommProtocol_Status_Response_Format Status Response Message Format Constants
  * @{
  */
//---------------------------------------------------------------------------------------------------
#define STATUS_RESPONSE_HEALTH_STATUS_INDEX         0
#define STATUS_RESPONSE_HEALTH_STATUS_LENGTH        4
#define STATUS_RESPONSE_CAN_RXD_ERROR_INDEX         (STATUS_RESPONSE_HEALTH_STATUS_INDEX+STATUS_RESPONSE_HEALTH_STATUS_LENGTH)
#define STATUS_RESPONSE_CAN_RXD_ERROR_LENGTH        4
#define STATUS_RESPONSE_CAN_TXD_ERROR_INDEX         (STATUS_RESPONSE_CAN_RXD_ERROR_INDEX+STATUS_RESPONSE_CAN_RXD_ERROR_LENGTH)
#define STATUS_RESPONSE_CAN_TXD_ERROR_LENGTH        4
#define STATUS_RESPONSE_CAN_BUS_OFF_INDEX           (STATUS_RESPONSE_CAN_TXD_ERROR_INDEX+STATUS_RESPONSE_CAN_TXD_ERROR_LENGTH)
#define STATUS_RESPONSE_CAN_BUS_OFF_LENGTH          4
#define STATUS_RESPONSE_CONVERTER_TYPE_INDEX        (STATUS_RESPONSE_CAN_BUS_OFF_INDEX+STATUS_RESPONSE_CAN_BUS_OFF_LENGTH)
#define STATUS_RESPONSE_CONVERTER_TYPE_LENGTH       1
#define STATUS_RESPONSE_SUPPORTED_FEATURES_INDEX    (STATUS_RESPONSE_CONVERTER_TYPE_INDEX+STATUS_RESPONSE_CONVERTER_TYPE_LENGTH)
#define STATUS_RESPONSE_SUPPORTED_FEATURES_LENGTH   4
#define STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_INDEX         (STATUS_RESPONSE_SUPPORTED_FEATURES_INDEX+STATUS_RESPONSE_SUPPORTED_FEATURES_LENGTH)
#define STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_LENGTH        1
#define STATUS_RESPONSE_MAIN_FILTER_CHANNEL_ID_SET_INDEX        (STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_INDEX+STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_LENGTH)
#define STATUS_RESPONSE_MAIN_FILTER_CHANNEL_ID_SET_LENGTH       4
#define STATUS_RESPONCE_NUMBER_OF_ADDITIONAL_FILTERS_INDEX (STATUS_RESPONSE_MAIN_FILTER_CHANNEL_ID_SET_INDEX+STATUS_RESPONSE_MAIN_FILTER_CHANNEL_ID_SET_LENGTH)
#define STATUS_RESPONCE_NUMBER_OF_ADDITIONAL_FILTERS_LENGTH 1
#define STATUS_RESPONSE_MESSAGE_DATA_LENGTH         (STATUS_RESPONSE_HEALTH_STATUS_LENGTH +\
                                                     STATUS_RESPONSE_CAN_RXD_ERROR_LENGTH  +\
                                                     STATUS_RESPONSE_CAN_TXD_ERROR_LENGTH  +\
                                                     STATUS_RESPONSE_CAN_BUS_OFF_LENGTH    +\
                                                     STATUS_RESPONSE_CONVERTER_TYPE_LENGTH +\
                                                     STATUS_RESPONSE_SUPPORTED_FEATURES_LENGTH+\
                                                     STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_LENGTH+\
                                                     STATUS_RESPONSE_MAIN_FILTER_CHANNEL_ID_SET_LENGTH+\
                                                     STATUS_RESPONCE_NUMBER_OF_ADDITIONAL_FILTERS_LENGTH)
/**
  * @}
  */
//---------------------------------------------------------------------------------------------------
/** @defgroup CommProtocol_CAN_FD_Frame_Format CAN FD Frame Format Constants
  * @{
  */
//---------------------------------------------------------------------------------------------------
#define CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_INDEX      0
#define CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_LENGTH     2
#define CAN_FD_FRAME_CHANNEL_GROUP_INDEX                (CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_INDEX+CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_LENGTH)
#define CAN_FD_FRAME_CHANNEL_GROUP_LENGTH               1
#define CAN_FD_FRAME_CHANNEL_ID_SET_INDEX               (CAN_FD_FRAME_CHANNEL_GROUP_INDEX+CAN_FD_FRAME_CHANNEL_GROUP_LENGTH)
#define CAN_FD_FRAME_CHANNEL_ID_SET_LENGTH              4
#define CAN_FD_FRAME_TIMESTAMP_INDEX                    (CAN_FD_FRAME_CHANNEL_ID_SET_INDEX+CAN_FD_FRAME_CHANNEL_ID_SET_LENGTH)
#define CAN_FD_FRAME_TIMESTAMP_LENGTH                   4
#define CAN_FD_FRAME_FLAGS_INDEX                        (CAN_FD_FRAME_TIMESTAMP_INDEX+CAN_FD_FRAME_TIMESTAMP_LENGTH)
#define CAN_FD_FRAME_FLAGS_LENGTH                       1
#define CAN_FD_FRAME_LENGTH_INDEX                       (CAN_FD_FRAME_FLAGS_INDEX+CAN_FD_FRAME_FLAGS_LENGTH)
#define CAN_FD_FRAME_LENGTH_LENGTH                      1
#define CAN_FD_FRAME_ID_INDEX                           (CAN_FD_FRAME_LENGTH_INDEX+CAN_FD_FRAME_LENGTH_LENGTH)
#define CAN_FD_FRAME_ID_LENGTH                          4
#define CAN_FD_FRAME_DATA_INDEX                         (CAN_FD_FRAME_ID_INDEX+CAN_FD_FRAME_ID_LENGTH)
#define CAN_FD_FRAME_NO_DATA_LENGTH                     (CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_LENGTH +\
                                                         CAN_FD_FRAME_CHANNEL_GROUP_LENGTH  +\
                                                         CAN_FD_FRAME_CHANNEL_ID_SET_LENGTH  +\
                                                         CAN_FD_FRAME_TIMESTAMP_LENGTH    +\
                                                         CAN_FD_FRAME_FLAGS_LENGTH +\
                                                         CAN_FD_FRAME_LENGTH_LENGTH +\
                                                         CAN_FD_FRAME_ID_LENGTH)
/**
  * @}
  */

/**
  * @}
  */
/* Private typedef -----------------------------------------------------------*/
/** @defgroup CommProtocol_Private_Types Private Types
  * @{
  */

/**
  * @brief Time Stamp Length ID for CAN frames in CAN and Notification Stream. Deprecated in SSP v3.0.0
  */
typedef enum
{
    NO_TIME_STAMP=0,
    ONE_BYTE_TIME_STAMP,
    TWO_BYTE_TIME_STAMP,
    FOUR_BYTE_TIME_STAMP
}
TIME_STAMP_LENGTH_ID;

/**
  * @brief Communication Protocol Message IDs
  */
typedef enum
{
    MESSAGE_ID_UNDEF=0,
    MESSAGE_ID_CAN_AND_NOTIFICATION_STREAM,
    MESSAGE_ID_STATUS_REQUEST,
    MESSAGE_ID_STATUS_RESPONSE,
    MESSAGE_ID_HEARTBEAT,
    MESSAGE_ID_CAN_FD_STREAM
}
COMM_PROTOCOL_MESSAGE_ID;
/**
  * @}
  */
/* Private macros ------------------------------------------------------------*/
/** @defgroup CommProtocol_Private_Macros Private Macros
  * @{
  */
#define SetFlag(Flags,FLAG_ID) ((Flags)|=(FLAG_ID))
#define IsFlagSet(Flags,FLAG_ID) (((Flags) & (FLAG_ID)) == (FLAG_ID))
/**
  * @}
  */
/* Private function prototypes -----------------------------------------------*/
/** @defgroup CommProtocol_Private_Function_Prototypes Private Function Prototypes
  * @{
  */
BOOL_t CPParseCommNodeFilters(const PM_PROTOCOL_MESSAGE_t *pPMessage,WORD_t wFirstFilterIndex,CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);
void CPAddCommNodeFilters(PM_PROTOCOL_MESSAGE_t *pPMessage,WORD_t wFirstFilterIndex,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList);
/* Private variables ---------------------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Imported variables --------------------------------------------------------*/
/* Global function definitions------------------------------------------------*/
/** @addtogroup CommProtocol_Exported_Functions
  * @{
  */

/** @addtogroup CommProtocol_Exported_Deprecated_Functions
  * @{
  */

/**
  * @brief Parses CAN and Notification Stream. Deprecated in SSP v3.0.0
  *
  * @note The parser is called on the protocol message. The callback function is
  *       called when either CAN or Notification frame is ready. The callback functions
  *       can be called multiple times, since the protocol message can contain more
  *       than one CAN or Notification frame.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[in] OnCanFrameParsed: callback function called on the successful parsing
  *            of the CAN frame. The function is provided with a pointer to the parsed
  *            CAN frame structure and with the arg pointer. Can be NULL if not used.
  * @param[in] OnNotificationFrameParsed: callback function called on the successful
  *            parsing of the Notification frame. The function is provided with a pointer
  *            to the parsed Notification frame structure and with the arg pointer. Can be NULL
  *            if not used.
  * @param[in] arg: pointer to an arbitrary data passed to OnCanFrameParsed() and
  *            OnNotificationFrameParsed() functions. Can be NULL if not used.
  *
  * @retval Returns CT_TRUE if the protocol message contains CAN and Notification Stream.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPParseCANDataAndNotificationStream(const PM_PROTOCOL_MESSAGE_t *pPMessage,
                CP_ON_CAN_FRAME_PARSED_FUNCTION_t OnCanFrameParsed,
                CP_ON_NOTIFICATION_FRAME_PARSED_FUNCTION_t OnNotificationFrameParsed,
                void *arg)
{
    CP_CAN_FRAME_t CANFrame;
    CP_NOTIFICATION_FRAME_t NotificationFrame;
    int i;
    int iDataPos;
    BYTE_t bFirstByte;
    BOOL_t bEIdFlag,bRemFrameFlag;
    TIME_STAMP_LENGTH_ID TimeStampLengthId;
    BYTE_t bTimeStampLength;
    DWORD_t dwTimeStamp,dwId;
    BYTE_t bIdLength,bLength;
    BYTE_t bNotificationId;
    DWORD_t dwNotificationData;

    if(pPMessage->wProtocolID!=COMM_PROTOCOL_ID) return CT_FALSE;
    if(pPMessage->wMessageID!=MESSAGE_ID_CAN_AND_NOTIFICATION_STREAM) return CT_FALSE;

    iDataPos=0;
    while(iDataPos<pPMessage->wDataLength)
    {
        bFirstByte=pPMessage->Data[iDataPos++];
        if(IsFlagSet(bFirstByte,FIRST_BYTE_NOTIFICATION_FRAME_FLAG))
        {
            // Notification Frame
            bNotificationId=bFirstByte & 0x7f;
            dwNotificationData=0;
            for(i=0;i<(NOTIFICATION_FRAME_LENGTH-1);i++)
            {
                dwNotificationData|=((DWORD_t)pPMessage->Data[iDataPos++])<<(i*8);
            }
            memset(&NotificationFrame,0,sizeof(NotificationFrame));
            NotificationFrame.bId=bNotificationId;
            NotificationFrame.dwData=dwNotificationData;
            if(iDataPos<=pPMessage->wDataLength)
            {
                if(OnNotificationFrameParsed!=NULL) OnNotificationFrameParsed(&NotificationFrame,arg);
            }
            continue;
        }
        // CAN Data Frame
        bLength=bFirstByte & 0x0f;
        if(bLength> CP_CAN_MAX_DATA_LENGTH) bLength=CP_CAN_MAX_DATA_LENGTH;
        bEIdFlag=((bFirstByte>>4) & 0x01)==0x01;

        TimeStampLengthId=(TIME_STAMP_LENGTH_ID)((bFirstByte>>5) & 0x03);
        if(TimeStampLengthId==FOUR_BYTE_TIME_STAMP)  bTimeStampLength=4;
        else if(TimeStampLengthId==TWO_BYTE_TIME_STAMP)  bTimeStampLength=2;
        else if(TimeStampLengthId==ONE_BYTE_TIME_STAMP)  bTimeStampLength=1;
        else bTimeStampLength=0;

        dwTimeStamp=0;
        for(i=0;i<bTimeStampLength;i++)
        {
            dwTimeStamp|=((DWORD_t)pPMessage->Data[iDataPos++])<<(i*8);
        }

        bIdLength=bEIdFlag ? 4 : 2;
        dwId=0;
        for(i=0;i<bIdLength;i++)
        {
            dwId|=((DWORD_t)pPMessage->Data[iDataPos++])<<(i*8);
        }

        if(bEIdFlag)
        {
            bRemFrameFlag=(dwId & RF_EID_FLAG)==RF_EID_FLAG;
            dwId&=EID_MASK;
        }
        else
        {
            bRemFrameFlag=(dwId & RF_ID_FLAG)==RF_ID_FLAG;
            dwId&=ID_MASK;
        }

        memset(&CANFrame,0,sizeof(CANFrame));
        CANFrame.dwId=dwId;
        CANFrame.bEIdFlag=bEIdFlag;
        CANFrame.bRemFrameFlag=bRemFrameFlag;
        CANFrame.bLength=bLength;
        CANFrame.dwTimeStamp=dwTimeStamp;
        if(!bRemFrameFlag)
        {
            for(i=0;i<bLength;i++) CANFrame.Data[i]=pPMessage->Data[iDataPos++];
        }
        if(iDataPos<=pPMessage->wDataLength)
        {
            if(OnCanFrameParsed!=NULL) OnCanFrameParsed(&CANFrame,arg);
        }
    }
    return CT_TRUE;
}

/**
  * @brief Prepares an empty CAN and Notification Stream. Deprecated in SSP v3.0.0
  *
  * @note This function should be called prior to any call to
  *       the CPAddCANFrame() or CPAddNotificationFrame() functions.
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval None
  */
void CPPrepareCANDataAndNotificationStream(PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=COMM_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_CAN_AND_NOTIFICATION_STREAM;
}

/**
  * @brief Adds CAN frame to the CAN and Notification Stream. Deprecated in SSP v3.0.0
  *
  * @note An empty CAN and Notification Stream message should be prepared using the
  *       CPPrepareCANDataAndNotificationStream() function before calling this
  *       function for the first time.
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in]  pCANFrame: pointer to the CAN frame structure.
  *
  * @retval Returns CT_TRUE if the CAN frame was added successfully. Otherwise,
  *         returns CT_FALSE.
  */
BOOL_t CPAddCANFrame(PM_PROTOCOL_MESSAGE_t *pPMessage,const CP_CAN_FRAME_t *pCANFrame)
{
    int i;
    BYTE_t bFirstByte;
    BOOL_t bEIdFlag,bRemFrameFlag;
    TIME_STAMP_LENGTH_ID TimeStampLengthId;
    BYTE_t bTimeStampLength;
    DWORD_t dwTimeStamp,dwId;
    BYTE_t bIdLength,bLength;
    BYTE_t bDataLength,bRecordLength;

    dwId=pCANFrame->dwId;
    dwTimeStamp=pCANFrame->dwTimeStamp;
    bLength=pCANFrame->bLength;
    bEIdFlag=pCANFrame->bEIdFlag;
    bRemFrameFlag=pCANFrame->bRemFrameFlag;

    if(dwTimeStamp>0xffff)
    {
        TimeStampLengthId=FOUR_BYTE_TIME_STAMP;
        bTimeStampLength=4;
    }
    else if(dwTimeStamp>0xff)
    {
        TimeStampLengthId=TWO_BYTE_TIME_STAMP;
        bTimeStampLength=2;
    }
    else if(dwTimeStamp>0)
    {
        TimeStampLengthId=ONE_BYTE_TIME_STAMP;
        bTimeStampLength=1;
    }
    else
    {
        TimeStampLengthId=NO_TIME_STAMP;
        bTimeStampLength=0;
    }
    bIdLength=bEIdFlag ? 4 : 2;
    if(bLength> CP_CAN_MAX_DATA_LENGTH) bLength= CP_CAN_MAX_DATA_LENGTH;
    bDataLength=bRemFrameFlag ? 0 : bLength;
    bRecordLength=1+bTimeStampLength+bIdLength+bDataLength;
    if((pPMessage->wDataLength+bRecordLength)>sizeof(pPMessage->Data)) return CT_FALSE; // No Space to write this frame
    //---Write CAN Frame to Buffer ---------------------------------------
    bFirstByte=(bLength & 0x0f)| ((TimeStampLengthId & 0x03)<<5);
    if(bEIdFlag) bFirstByte|=FIRST_BYTE_EID_FLAG;
    pPMessage->Data[pPMessage->wDataLength++]=bFirstByte;
    for(i=0;i<bTimeStampLength;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(dwTimeStamp & 0xff);
        dwTimeStamp>>=8;
    }
    if(bRemFrameFlag)
    {
        dwId|=(DWORD_t)(bEIdFlag ? RF_EID_FLAG : RF_ID_FLAG);
    }
    for(i=0;i<bIdLength;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(dwId & 0xff);
        dwId>>=8;
    }
    if(!bRemFrameFlag)
    {
        for(i=0;i<bLength;i++) pPMessage->Data[pPMessage->wDataLength++]=pCANFrame->Data[i];
    }
    return CT_TRUE;
}

/**
  * @brief Adds Notification frame to the CAN and Notification Stream. Deprecated in SSP v3.0.0
  *
  * @note An empty CAN and Notification Stream message should be prepared using the
  *       CPPrepareCANDataAndNotificationStream() function before calling this
  *       function for the first time.
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in]  pNotificationFrame: pointer to the Notification frame structure.
  *
  * @retval Returns CT_TRUE if the frame was added successfully. Otherwise,
  *         returns CT_FALSE.
  */
BOOL_t CPAddNotificationFrame(PM_PROTOCOL_MESSAGE_t *pPMessage,const CP_NOTIFICATION_FRAME_t *pNotificationFrame)
{
    BYTE_t bId;
    DWORD_t dwData;
    BYTE_t bFirstByte;
    int i;

    bId=pNotificationFrame->bId;
    dwData=pNotificationFrame->dwData;

    if(!CPIsNotificationFrameFit(pPMessage)) return CT_FALSE; // No Space to write this frame

    bFirstByte=(bId & 0x7f) | FIRST_BYTE_NOTIFICATION_FRAME_FLAG;
    pPMessage->Data[pPMessage->wDataLength++]=bFirstByte;
    for(i=0;i<4;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(dwData & 0xff);
        dwData>>=8;
    }
    return CT_TRUE;
}

/**
  * @brief Checks whether there is enough space to write a CAN frame
  *        into the CAN and Notification Stream. Deprecated in SSP v3.0.0
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval Returns CT_TRUE if a CAN frame will fit into the CAN and Notification Stream.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPIsCANFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    if((pPMessage->wDataLength+CAN_FRAME_LENGTH_MAX)>sizeof(pPMessage->Data)) return CT_FALSE; // No Space to write this frame
    return CT_TRUE;
}

/**
  * @brief Checks whether there is enough space to write a Notification frame
  *        into the CAN and Notification Stream. Deprecated in SSP v3.0.0
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval Returns CT_TRUE if a Notification frame will fit into the CAN and Notification Stream.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPIsNotificationFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    if((pPMessage->wDataLength+NOTIFICATION_FRAME_LENGTH)>sizeof(pPMessage->Data)) return CT_FALSE; // No Space to write this frame
    return CT_TRUE;
}
/**
  * @}
  */

/** @addtogroup CommProtocol_Exported_CAN_FD_Stream_Functions
  * @{
  */
/**
  * @brief Parses CAN FD Stream
  *
  * @note The parser is called on the protocol message. The callback function is
  *       called when a CAN FD frame is ready. The CAN FD frame can also
  *       contain a Classical CAN frame. The callback function can be called
  *       multiple times, since the protocol message can contain more than one CAN FD
  *       frame, unless the node requested one frame per message feature in
  *       CP_COMMUNICATION_NODE_PROPERTY_DATA_t.
  *
  *       The parser was added in SSP v3.0.0 to replace deprecated parser for
  *       CAN and Notification Stream CPParseCANDataAndNotificationStream().
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[in] OnCANFDFrameParsed: callback function called on the successful parsing
  *            of the CAN FD frame. The function is provided with a pointer to the parsed
  *            CAN FD frame structure (that can also contain a Classical CAN frame), a
  *            pointer to the CAN frame routing data structure, an absolute timestamp, and
  *            the arg pointer.
  *            Can be NULL if not used.
  * @param[in] arg: pointer to an arbitrary data passed to OnCANFDFrameParsed() function.
  *            Can be NULL if not used.
  *
  * @retval Returns CT_TRUE if the protocol message contains CAN FD Stream.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPParseCANFDStream(const PM_PROTOCOL_MESSAGE_t *pPMessage,CP_ON_CAN_FD_FRAME_PARSED_FUNCTION_t OnCANFDFrameParsed,void *arg)
{
    int iDataPos;
    CP_CAN_FD_FRAME_t CANFDFrame;
    CP_CAN_FRAME_ROUTING_DATA_t CANFrameRoutingData;
    BYTE_t bChannelGroup;
    WORD_t wPhysicalChannelNumber;
    WORD_t wFlags;
    DWORD_t dwChannelIDSet;
    BYTE_t bCANFDFlags;
    DWORD_t dwAbsTimeStamp,dwID;
    BYTE_t bCANFDLength;
    int i;

    if(pPMessage->wProtocolID!=COMM_PROTOCOL_ID) return CT_FALSE;
    if(pPMessage->wMessageID!=MESSAGE_ID_CAN_FD_STREAM) return CT_FALSE;
    iDataPos=0;
    while(iDataPos<pPMessage->wDataLength)
    {
        wPhysicalChannelNumber=0;
        for(i=0;i<CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_LENGTH;i++)
        {
            wPhysicalChannelNumber|=((WORD_t)pPMessage->Data[iDataPos++])<<(i*8);
        }
        wFlags=wPhysicalChannelNumber & ~(PHYSICAL_CHANNEL_NUMBER_MASK);
        wPhysicalChannelNumber&=PHYSICAL_CHANNEL_NUMBER_MASK;

        bChannelGroup=pPMessage->Data[iDataPos++];
        dwChannelIDSet=0;
        for(i=0;i<CAN_FD_FRAME_CHANNEL_ID_SET_LENGTH;i++)
        {
            dwChannelIDSet|=((DWORD_t)pPMessage->Data[iDataPos++])<<(i*8);
        }
        dwAbsTimeStamp=0;
        for(i=0;i<CAN_FD_FRAME_TIMESTAMP_LENGTH;i++)
        {
            dwAbsTimeStamp|=((DWORD_t)pPMessage->Data[iDataPos++])<<(i*8);
        }
        bCANFDFlags=pPMessage->Data[iDataPos++];
        bCANFDLength=pPMessage->Data[iDataPos++];
        if(bCANFDLength> CP_CAN_FD_MAX_DATA_LENGTH) bCANFDLength= CP_CAN_FD_MAX_DATA_LENGTH;
        dwID=0;
        for(i=0;i<CAN_FD_FRAME_ID_LENGTH;i++)
        {
            dwID|=((DWORD_t)pPMessage->Data[iDataPos++])<<(i*8);
        }

        memset(&CANFDFrame,0,sizeof(CANFDFrame));
        CANFDFrame.bFlags=bCANFDFlags;
        CANFDFrame.bLength=bCANFDLength;
        CANFDFrame.dwID=(dwID & FD_ID_MASK);
        if(IsFlagSet(bCANFDFlags,CP_CAN_FD_FLAG_ERROR_MESSAGE) || IsFlagSet(bCANFDFlags,CP_CAN_FD_FLAG_FD_FRAME) || !IsFlagSet(bCANFDFlags,CP_CAN_FD_FLAG_REMOTE_FRAME))
        {
            // Error Message, CAN FD frame or Classical Data frame
            for(i=0;i<CANFDFrame.bLength;i++)
            {
                CANFDFrame.Data[i]=pPMessage->Data[iDataPos++];
            }
        }
        if(iDataPos<=pPMessage->wDataLength)
        {
            if(OnCANFDFrameParsed!=NULL)
            {
                memset(&CANFrameRoutingData,0,sizeof(CANFrameRoutingData));
                CANFrameRoutingData.wFlags=wFlags;
                CANFrameRoutingData.wPhysicalChannelNumber=wPhysicalChannelNumber;
                CANFrameRoutingData.CANAddr.bChannelGroup=bChannelGroup;
                CANFrameRoutingData.CANAddr.dwChannelIDSet=dwChannelIDSet;

                OnCANFDFrameParsed(&CANFDFrame,&CANFrameRoutingData,dwAbsTimeStamp,arg);
            }
        }
    }
    return CT_TRUE;
}

/**
  * @brief Prepares an empty CAN FD Stream
  *
  * @note This function should be called prior to any call to
  *       the CPAddCANFrame() or CPAddNotificationFrame() functions.
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval None
  */
void CPPrepareCANFDStream(PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=COMM_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_CAN_FD_STREAM;
}

/**
  * @brief Adds a CAN FD frame to the CAN FD Stream
  *
  * @note An empty CAN FD Stream message should be prepared using the
  *       CPPrepareCANFDStream() function before calling this
  *       function for the first time.
  *       The function can also add Classical CAN frames to the CAN FD Stream.
  *
  *       The function was added in SSP v3.0.0 to replace deprecated CPAddCANFrame()
  *       function used to add Classical CAN frames to CAN and Notification Stream.
..*
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in]  pCANFDFrame: pointer to the CAN FD frame structure. The structure can contain a
  *             Classical CAN frame.
  * @param[in]  pFrameRoutingData: pointer to the CAN FD Frame routing data structure.
  * @param[in]  dwAbsoluteTimeStamp: pointer to the CAN FD Frame routing data structure.
  *
  * @retval Returns CT_TRUE if the CAN frame was added successfully. Otherwise,
  *         returns CT_FALSE.
  */
BOOL_t CPAddCANFDFrame(PM_PROTOCOL_MESSAGE_t *pPMessage,const CP_CAN_FD_FRAME_t *pCANFDFrame,const CP_CAN_FRAME_ROUTING_DATA_t *pCANFrameRoutingData,DWORD_t dwAbsoluteTimeStamp)
{
    BYTE_t bRecordLength;
    BYTE_t bDataLength;
    WORD_t wPhysicalChannelNumber;
    DWORD_t dwID;
    DWORD_t dwChannelIDSet;
    BOOL_t bIsAddData;
    int i;

    bRecordLength=CAN_FD_FRAME_NO_DATA_LENGTH;
    bDataLength=pCANFDFrame->bLength;
    if(bDataLength> CP_CAN_FD_MAX_DATA_LENGTH) bDataLength= CP_CAN_FD_MAX_DATA_LENGTH;
    bIsAddData=CT_FALSE;
    if(IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_ERROR_MESSAGE) || IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_FD_FRAME) || !IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_REMOTE_FRAME))
    {
        // Error Frame, FD Frame or Classical Data Frame
        bRecordLength+=bDataLength;
        bIsAddData=CT_TRUE;
    }
    if((pPMessage->wDataLength + bRecordLength)>sizeof(pPMessage->Data)) return CT_FALSE; // No Space to write this frame
    //---Write CAN Frame to Buffer ---------------------------------------

    wPhysicalChannelNumber=(pCANFrameRoutingData->wPhysicalChannelNumber & PHYSICAL_CHANNEL_NUMBER_MASK)|(pCANFrameRoutingData->wFlags & ~(PHYSICAL_CHANNEL_NUMBER_MASK));
    for(i=0;i<CAN_FD_FRAME_PHYSICAL_CHANNEL_NUMBER_LENGTH;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(wPhysicalChannelNumber & 0xff);
        wPhysicalChannelNumber>>=8;
    }

    pPMessage->Data[pPMessage->wDataLength++]=pCANFrameRoutingData->CANAddr.bChannelGroup;

    dwChannelIDSet=pCANFrameRoutingData->CANAddr.dwChannelIDSet;
    for(i=0;i<CAN_FD_FRAME_CHANNEL_ID_SET_LENGTH;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(dwChannelIDSet & 0xff);
        dwChannelIDSet>>=8;
    }

    for(i=0;i<CAN_FD_FRAME_TIMESTAMP_LENGTH;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(dwAbsoluteTimeStamp & 0xff);
        dwAbsoluteTimeStamp>>=8;
    }
    pPMessage->Data[pPMessage->wDataLength++]=pCANFDFrame->bFlags;
    pPMessage->Data[pPMessage->wDataLength++]=bDataLength;

    dwID=(pCANFDFrame->dwID & FD_ID_MASK);
    for(i=0;i<CAN_FD_FRAME_ID_LENGTH;i++)
    {
        pPMessage->Data[pPMessage->wDataLength++]=(BYTE_t)(dwID & 0xff);
        dwID>>=8;
    }
    if(bIsAddData)
    {
        // Error Message, FD Frame or Classical Data Frame
        for(i=0;i<bDataLength;i++)
        {
            pPMessage->Data[pPMessage->wDataLength++]=pCANFDFrame->Data[i];
        }
    }
    return CT_TRUE;
}

/**
  * @brief Checks whether there is enough space to write a CAN FD frame
  *        into the CAN FD Stream.
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval Returns CT_TRUE if a CAN frame will fit into the CAN FD Stream.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPIsCANFDFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    if((pPMessage->wDataLength+(CAN_FD_FRAME_NO_DATA_LENGTH + CP_CAN_FD_MAX_DATA_LENGTH))>sizeof(pPMessage->Data)) return CT_FALSE; // No Space to write this frame
    return CT_TRUE;
}

/**
  * @brief Checks whether there is enough space to write a Classical CAN frame 
  *        into the CAN FD Stream.
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval Returns CT_TRUE if a CAN frame will fit into the CAN FD Stream.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPIsCANClassicalCANFDFrameFit(const PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    if((pPMessage->wDataLength+(CAN_FD_FRAME_NO_DATA_LENGTH + CP_CAN_MAX_DATA_LENGTH))>sizeof(pPMessage->Data)) return CT_FALSE; // No Space to write this frame
    return CT_TRUE;
}

/**
  * @}
  */

/**
  * @brief Parses the Status Request.
  *
  * @note The parser is called on the protocol message.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  *
  * @retval Returns CT_TRUE if the protocol message contains the Status Request.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPParseStatusRequest(const PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    if(pPMessage->wProtocolID!=COMM_PROTOCOL_ID) return CT_FALSE;
    return (pPMessage->wMessageID==MESSAGE_ID_STATUS_REQUEST);
}

/**
  * @brief Parses Status Response message.
  *
  * @note The parser is called on the protocol message. It returns the values of the Status
  *       Response fields.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[out] pdwHealthData: pointer to the Health Data. It contains the health status
  *                            information of the converter and its subsystems.
  * @param[out] pdwCANRxDErrors: pointer to the CAN Receive Error Counter.
  * @param[out] pdwCANTxDErrors: pointer to the CAN Transmit Error Counter.
  * @param[out] pdwCANBusOffErrors: pointer to the CAN Bus Off Counter.
  * @param[out] pCommNodeSupportedFeatures: pointer to the Communication Node Supported Features.
  * @param[out] pCommNodeFilterList: pointer to the Communication Node Filter List structure. Can be NULL if not used
  *
  * @retval Returns CT_TRUE if the protocol message contains the Status Response.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPParseStatusResponse(const PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t *pdwHealthData,DWORD_t *pdwCANRxDErrors,DWORD_t *pdwCANTxDErrors,
                            DWORD_t *pdwCANBusOffErrors,CP_CONVERTER_TYPE_t *pConverterType, DWORD_t *pCommNodeSupportedFeatures,
                            CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    if(pPMessage->wProtocolID!=COMM_PROTOCOL_ID) return CT_FALSE;
    if(pPMessage->wMessageID!=MESSAGE_ID_STATUS_RESPONSE) return CT_FALSE;

    if(pPMessage->bMessageVersion == 0)
    {
        if(pPMessage->wDataLength<STATUS_RESPONSE_CONVERTER_TYPE_INDEX) return CT_FALSE;
    }
    else if(pPMessage->bMessageVersion == 1)
    {
        if(pPMessage->wDataLength<STATUS_RESPONSE_SUPPORTED_FEATURES_INDEX) return CT_FALSE;
    }
    else if(pPMessage->bMessageVersion == 2)
    {
        if(pPMessage->wDataLength<STATUS_RESPONCE_NUMBER_OF_ADDITIONAL_FILTERS_INDEX) return CT_FALSE;
    }
    else
    {
        // pPMessage->bMessageVersion >=3
        if(pPMessage->wDataLength<STATUS_RESPONSE_MESSAGE_DATA_LENGTH) return CT_FALSE;
    }

    *pdwHealthData=PMGetDataFromMessage(pPMessage,STATUS_RESPONSE_HEALTH_STATUS_INDEX,STATUS_RESPONSE_HEALTH_STATUS_LENGTH);
    *pdwCANRxDErrors=PMGetDataFromMessage(pPMessage,STATUS_RESPONSE_CAN_RXD_ERROR_INDEX,STATUS_RESPONSE_CAN_RXD_ERROR_LENGTH);
    *pdwCANTxDErrors=PMGetDataFromMessage(pPMessage,STATUS_RESPONSE_CAN_TXD_ERROR_INDEX,STATUS_RESPONSE_CAN_TXD_ERROR_LENGTH);
    *pdwCANBusOffErrors=PMGetDataFromMessage(pPMessage,STATUS_RESPONSE_CAN_BUS_OFF_INDEX,STATUS_RESPONSE_CAN_BUS_OFF_LENGTH);
    *pConverterType=(pPMessage->bMessageVersion >= 1)? (CP_CONVERTER_TYPE_t)pPMessage->Data[STATUS_RESPONSE_CONVERTER_TYPE_INDEX] : CP_CONVERTER_CAN_ETHERNET;
    *pCommNodeSupportedFeatures = (pPMessage->bMessageVersion >= 2)? PMGetDataFromMessage(pPMessage, STATUS_RESPONSE_SUPPORTED_FEATURES_INDEX, STATUS_RESPONSE_SUPPORTED_FEATURES_LENGTH) : 0;

    if(pCommNodeFilterList != NULL)
    {
        CPParseCommNodeFilters(pPMessage,STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_INDEX,pCommNodeFilterList);
    }
    return CT_TRUE;
}

/**
  * @brief Parses Heartbeat message.
  *
  * @note The parser is called on the protocol message. It returns the values of the Heartbeat fields.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[out] pdwMessageNumber: pointer to the Message Number.
  * @param[out] pdwTimeInterval: pointer to the Time Interval in milliseconds elapsed from
  *                              the last Heartbeat message.
  * @param[out] pdwHealthData: pointer to the converter Health Data.
  * @param[out] pConverterType: pointer to the Converter Type.
  * @param[out] pCommNodeSupportedFeatures: pointer to the Communication Node Supported Features.
  * @param[out] pCommNodeFilterList: pointer to the Communication Node Filter List structure. Can be NULL if not used.
  *
  * @retval Returns CT_TRUE if the protocol message contains the Heartbeat message.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t CPParseHeartbeat(const PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t *pdwMessageNumber,DWORD_t *pdwTimeInterval,DWORD_t *pdwHealthData,CP_CONVERTER_TYPE_t *pConverterType,
                        DWORD_t *pCommNodeSupportedFeatures,CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    if(pPMessage->wProtocolID!=COMM_PROTOCOL_ID) return CT_FALSE;
    if(pPMessage->wMessageID!=MESSAGE_ID_HEARTBEAT) return CT_FALSE;

    if(pPMessage->bMessageVersion == 0)
    {
        if(pPMessage->wDataLength<HEARTBEAT_CONVERTER_TYPE_INDEX) return CT_FALSE;
    }
    else if(pPMessage->bMessageVersion == 1)
    {
        if(pPMessage->wDataLength<HEARTBEAT_SUPPORTED_FEATURES_INDEX) return CT_FALSE;
    }
    else if(pPMessage->bMessageVersion == 2)
    {
        if(pPMessage->wDataLength<HEARTBEAT_NUMBER_OF_ADDITIONAL_FILTERS_INDEX) return CT_FALSE;
    }
    else
    {
        // pPMessage->bMessageVersion >=3
        if(pPMessage->wDataLength<HEARTBEAT_MESSAGE_DATA_LENGTH) return CT_FALSE;
    }

    *pdwMessageNumber=PMGetDataFromMessage(pPMessage,HEARTBEAT_MESSAGE_NUMBER_INDEX,HEARTBEAT_MESSAGE_NUMBER_LENGTH);
    *pdwTimeInterval=PMGetDataFromMessage(pPMessage,HEARTBEAT_TIME_INTERVAL_INDEX,HEARTBEAT_TIME_INTERVAL_LENGTH);
    *pdwHealthData=PMGetDataFromMessage(pPMessage,HEARTBEAT_HEALTH_DATA_INDEX,HEARTBEAT_HEALTH_DATA_LENGTH);
    *pConverterType=(pPMessage->bMessageVersion >= 1)? (CP_CONVERTER_TYPE_t)pPMessage->Data[HEARTBEAT_CONVERTER_TYPE_INDEX] : CP_CONVERTER_CAN_ETHERNET;
    *pCommNodeSupportedFeatures = (pPMessage->bMessageVersion >= 2)? PMGetDataFromMessage(pPMessage, HEARTBEAT_SUPPORTED_FEATURES_INDEX, HEARTBEAT_SUPPORTED_FEATURES_LENGTH) : 0;

    if(pCommNodeFilterList != NULL)
    {
        CPParseCommNodeFilters(pPMessage,HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_INDEX,pCommNodeFilterList);
    }
    return CT_TRUE;
}

/**
  * @brief Generates a Status Request Message.
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval None
  */
void CPGenStatusRequestMessage(PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=COMM_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_STATUS_REQUEST;
}

/**
  * @brief Generates a Status Response Message
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in] dwHealthData: Health Data. Contains the health status information of
  *                          the converter and its subsystems.
  * @param[in] dwCANRxDErrors: CAN Receive Error Counter.
  * @param[in] dwCANTxDErrors: CAN Transmit Error Counter.
  * @param[in] dwCANBusOffErrors: CAN Bus Off Counter.
  * @param[in] ConverterType: Converter Type.
  * @param[in] dwCommNodeSupportedFeatures: Communication Node Supported Features.
  * @param[in] pCommNodeFilterList: pointer to the Communication Node Filter List structure. Can be NULL if not used.
  *
  * @retval None
  */
void CPGenStatusResponseMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t dwHealthData,DWORD_t dwCANRxDErrors,DWORD_t dwCANTxDErrors,DWORD_t dwCANBusOffErrors,CP_CONVERTER_TYPE_t ConverterType,
                                DWORD_t dwCommNodeSupportedFeatures,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=COMM_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_STATUS_RESPONSE;
    pPMessage->bMessageVersion=3;

    pPMessage->wDataLength=STATUS_RESPONSE_MESSAGE_DATA_LENGTH;
    PMSetDataToMessage(pPMessage,STATUS_RESPONSE_HEALTH_STATUS_INDEX,STATUS_RESPONSE_HEALTH_STATUS_LENGTH,dwHealthData);
    PMSetDataToMessage(pPMessage,STATUS_RESPONSE_CAN_RXD_ERROR_INDEX,STATUS_RESPONSE_CAN_RXD_ERROR_LENGTH,dwCANRxDErrors);
    PMSetDataToMessage(pPMessage,STATUS_RESPONSE_CAN_TXD_ERROR_INDEX,STATUS_RESPONSE_CAN_TXD_ERROR_LENGTH,dwCANTxDErrors);
    PMSetDataToMessage(pPMessage,STATUS_RESPONSE_CAN_BUS_OFF_INDEX,STATUS_RESPONSE_CAN_BUS_OFF_LENGTH,dwCANBusOffErrors);
    pPMessage->Data[STATUS_RESPONSE_CONVERTER_TYPE_INDEX]=ConverterType;

    PMSetDataToMessage(pPMessage, STATUS_RESPONSE_SUPPORTED_FEATURES_INDEX, STATUS_RESPONSE_SUPPORTED_FEATURES_LENGTH, dwCommNodeSupportedFeatures);
    if (pCommNodeFilterList != NULL)
    {
        CPAddCommNodeFilters(pPMessage,STATUS_RESPONSE_MAIN_FILTER_CHANNEL_GROUP_INDEX,pCommNodeFilterList);
    }
}

/**
  * @brief Generates a Heartbeat Message.
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in] dwMessageNumber: Message Number.
  * @param[in] dwTimeInterval: Time Interval in milliseconds elapsed from the last
  *                            Heartbeat message.
  * @param[in] dwHealthData: Health Data, contains the health status information of
  *                          the converter and its subsystems.
  * @param[in] ConverterType: Converter Type.
  * @param[in] dwCommNodeSupportedFeatures: Communication Node Supported Features.
  * @param[in] pCommNodeFilterList: pointer to the Communication Node Filter List structure. Can be NULL if not used.
  *
  * @retval None
  */
void CPGenHearbeatMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,DWORD_t dwMessageNumber,DWORD_t dwTimeInterval,DWORD_t dwHealthData,CP_CONVERTER_TYPE_t ConverterType,
                        DWORD_t dwCommNodeSupportedFeatures,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=COMM_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_HEARTBEAT;
    pPMessage->bMessageVersion=3;

    pPMessage->wDataLength=HEARTBEAT_MESSAGE_DATA_LENGTH;
    PMSetDataToMessage(pPMessage,HEARTBEAT_MESSAGE_NUMBER_INDEX,HEARTBEAT_MESSAGE_NUMBER_LENGTH,dwMessageNumber);
    PMSetDataToMessage(pPMessage,HEARTBEAT_TIME_INTERVAL_INDEX,HEARTBEAT_TIME_INTERVAL_LENGTH,dwTimeInterval);
    PMSetDataToMessage(pPMessage,HEARTBEAT_HEALTH_DATA_INDEX,HEARTBEAT_HEALTH_DATA_LENGTH,dwHealthData);
    pPMessage->Data[HEARTBEAT_CONVERTER_TYPE_INDEX]=ConverterType;
    PMSetDataToMessage(pPMessage, HEARTBEAT_SUPPORTED_FEATURES_INDEX, HEARTBEAT_SUPPORTED_FEATURES_LENGTH, dwCommNodeSupportedFeatures);
    if (pCommNodeFilterList != NULL)
    {
        CPAddCommNodeFilters(pPMessage,HEARTBEAT_MAIN_FILTER_CHANNEL_GROUP_INDEX,pCommNodeFilterList);
    }
}

/** @addtogroup CommProtocol_Exported_Converting_Functions
  * @{
  */
/**
  * @brief Converts deprecated Classical CAN frame into CAN FD frame format.
  *
  *
  * @param[out] pCANFDFrame: pointer to the CAN FD frame.
  * @param[in] pCANFrame: pointer to the Classical CAN frame structure.
  *
  * @retval None.
  */
void CPConvertCANToCANFDFrame(CP_CAN_FD_FRAME_t *pCANFDFrame,const CP_CAN_FRAME_t *pCANFrame)
{
    int i;
    memset(pCANFDFrame,0,sizeof(*pCANFDFrame));

    pCANFDFrame->dwID=pCANFrame->dwId;
    pCANFDFrame->bLength=pCANFrame->bLength;
    if(pCANFDFrame->bLength>8) pCANFDFrame->bLength=8;
    if(pCANFrame->bEIdFlag) SetFlag(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_EXTENDED_ID);
    if(pCANFrame->bRemFrameFlag) SetFlag(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_REMOTE_FRAME);
    if(!pCANFrame->bRemFrameFlag)
    {
        for(i=0;i<pCANFDFrame->bLength;i++)
        {
            pCANFDFrame->Data[i]=pCANFrame->Data[i];
        }
    }
}

/**
  * @brief Converts CAN FD frame into deprecated Classical CAN frame format.
  *        CAN FD frames are converted into CAN Classical frames with data load truncated to the first 8 bytes.
  *        CAN Remote Frame flag set together with CAN FD flag is ignored.
  *
  * @param[out] pCANFrame: pointer to the Classical CAN frame structure.
  * @param[in] pCANFDFrame: pointer to the CAN FD frame structure.
  * @param[in] dwRelativeTimeStamp: relative Time Stamp (time interval between two consequent CAN frames).
  *
  * @retval Returns CT_TRUE if conversion is possible. Otherwise, returns CT_FALSE.
  */
BOOL_t CPConvertCANFDToCANrame(CP_CAN_FRAME_t *pCANFrame,const CP_CAN_FD_FRAME_t *pCANFDFrame,DWORD_t dwRelativeTimeStamp)
{
    int i;

    if(IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_ERROR_MESSAGE)) return CT_FALSE;

    memset(pCANFrame,0,sizeof(*pCANFrame));
    pCANFrame->dwId=pCANFDFrame->dwID;
    pCANFrame->bLength=pCANFDFrame->bLength>8 ? 8: pCANFDFrame->bLength;
    pCANFrame->bEIdFlag=IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_EXTENDED_ID);

    if(IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_FD_FRAME) || (!IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_REMOTE_FRAME)))
    {
        // FD frame or Classical Data frame
        // If Remote Frame flag is set in FD frame (incorrect setting), the Remote Frame flag is ignored and FD frame
        // is converted into Classical Data frame
        for(i=0;i<pCANFrame->bLength;i++)
        {
            pCANFrame->Data[i]=pCANFDFrame->Data[i];
        }
    }
    else
    {
        // Classical Remote frame
        pCANFrame->bRemFrameFlag=IsFlagSet(pCANFDFrame->bFlags,CP_CAN_FD_FLAG_REMOTE_FRAME);
    }

    pCANFrame->dwTimeStamp=dwRelativeTimeStamp;
    return CT_TRUE;
}

/**
  * @brief Checks whether a CAN Address passes the Communication Node Filter.
  *
  * @param[in] pCANAddr: pointer to the CAN Address structure.
  * @param[in] pCommNodeFilterList: pointer to the Communication Node Filter List structure.
  *
  * @retval Returns CT_TRUE if the CAN Address has passed the filter. Otherwise, returns CT_FALSE.
  */
BOOL_t CPIsCANAddressPassCommNodeFilter(const CP_CAN_ADDRESS_t *pCANAddr,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    int i;

    // Invalid CAN address should not pass the filter
    if(!CP_IsCANAddrValid(*pCANAddr)) return CT_FALSE;

    // If there are no filters, all CAN addresses will pass
    if(pCommNodeFilterList->bNumberOfFilters==0) return CT_TRUE;

    for(i=0;i<pCommNodeFilterList->bNumberOfFilters;i++)
    {
        // If the CAN address matches the filter it will pass
        if(CP_IsCANAddrMatch(*pCANAddr,pCommNodeFilterList->Filter[i])) return CT_TRUE;
    }
    return CT_FALSE;
}

/**
  * @}
  */
/**
  * @}
  */
/* Private function definitions-----------------------------------------------*/
/**
  * @}
  */

/**
  * @brief Parses Communication Node Filter List in Status Response and Heartbeat messages.
  *
  * @note The parser is called during parsing of the Status Response and Heartbeat protocol messages. It returns the value of the Communication Node Filter List
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[in] wFirstFilterIndex: index of the first Communication Node Filter location in the protocol message data.
  * @param[out] pCommNodeFilterList: pointer to the Communication Node Filter List structure. The structure is set to all 0 (filter is disabled) on unsuccessful function return.
  *
  * @retval Returns CT_TRUE if parsing was successful. Otherwise, returns CT_FALSE. 
  *
  */
BOOL_t CPParseCommNodeFilters(const PM_PROTOCOL_MESSAGE_t *pPMessage,WORD_t wFirstFilterIndex,CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    int i,iIndex;
    CP_CAN_ADDRESS_t CANAddress;

    memset(pCommNodeFilterList, 0, sizeof(*pCommNodeFilterList));

    if(pPMessage->bMessageVersion < 2) return CT_TRUE;
    // pPMessage->bMessageVersion >=2

    iIndex=wFirstFilterIndex;
    memset(&CANAddress,0,sizeof(CANAddress));
    CANAddress.bChannelGroup=pPMessage->Data[iIndex];
    CANAddress.dwChannelIDSet=PMGetDataFromMessage(pPMessage, iIndex+CP_CAN_ADDRESS_CHANNEL_GROUP_LENGTH, CP_CAN_ADDRESS_CHANNEL_ID_SET_LENGTH);
    if(!CP_IsCANAddrNULL(CANAddress))
    {
        pCommNodeFilterList->Filter[0]=CANAddress;
        pCommNodeFilterList->bNumberOfFilters=1;
        return CT_TRUE;
    }
    if(pPMessage->bMessageVersion==2) return CT_TRUE;
    iIndex+=CP_CAN_ADDRESS_LENGTH;
    pCommNodeFilterList->bNumberOfFilters=pPMessage->Data[iIndex++];

    if(pCommNodeFilterList->bNumberOfFilters==0) return CT_TRUE;
    if(pCommNodeFilterList->bNumberOfFilters==1 || pCommNodeFilterList->bNumberOfFilters>10 || 
        pPMessage->wDataLength<(iIndex+pCommNodeFilterList->bNumberOfFilters*CP_CAN_ADDRESS_LENGTH))
    {
        memset(pCommNodeFilterList, 0, sizeof(*pCommNodeFilterList));
        return CT_FALSE;
    }
    for(i=0;i<pCommNodeFilterList->bNumberOfFilters;i++)
    {
        memset(&CANAddress,0,sizeof(CANAddress));
        CANAddress.bChannelGroup=pPMessage->Data[iIndex];
        CANAddress.dwChannelIDSet=PMGetDataFromMessage(pPMessage, iIndex+CP_CAN_ADDRESS_CHANNEL_GROUP_LENGTH, CP_CAN_ADDRESS_CHANNEL_ID_SET_LENGTH);
        pCommNodeFilterList->Filter[i]=CANAddress;
        iIndex+=CP_CAN_ADDRESS_LENGTH;
    }
    return CT_TRUE;
}

/**
  * @brief Adds Communication Node Filter data to the Status Response and Heartbeat messages.
  *
  * @note This function is called during generation of the Status Response and Heartbeat protocol messages.
  *
  * @param[in,out] pPMessage: pointer to the protocol message structure.
  * @param[in] wFirstFilterIndex: index of the first Communication Node Filter location in the protocol message data.
  * @param[in] pCommNodeFilterList: pointer to the Communication Node Filter List structure.
  *
  * @retval None. 
  *
  */
void CPAddCommNodeFilters(PM_PROTOCOL_MESSAGE_t *pPMessage,WORD_t wFirstFilterIndex,const CP_COMMUNICATION_NODE_FILTER_LIST_t *pCommNodeFilterList)
{
    int i,iIndex;
    BYTE_t bNumberOfFilters;

    iIndex=wFirstFilterIndex;
    bNumberOfFilters=pCommNodeFilterList->bNumberOfFilters<=MAX_NUMBER_OF_COMM_NODE_FILTERS ? pCommNodeFilterList->bNumberOfFilters : MAX_NUMBER_OF_COMM_NODE_FILTERS;

    if(bNumberOfFilters==0) return;
    if(bNumberOfFilters==1)
    {
        pPMessage->Data[iIndex]=pCommNodeFilterList->Filter[0].bChannelGroup;
        PMSetDataToMessage(pPMessage, iIndex+CP_CAN_ADDRESS_CHANNEL_GROUP_LENGTH, CP_CAN_ADDRESS_CHANNEL_ID_SET_LENGTH, pCommNodeFilterList->Filter[0].dwChannelIDSet);
        return;
    }
    iIndex+=CP_CAN_ADDRESS_LENGTH;
    pPMessage->Data[iIndex++]=bNumberOfFilters;
    pPMessage->wDataLength+=bNumberOfFilters*CP_CAN_ADDRESS_LENGTH;

    for(i=0;i<bNumberOfFilters;i++)
    {
        pPMessage->Data[iIndex]=pCommNodeFilterList->Filter[i].bChannelGroup;
        PMSetDataToMessage(pPMessage, iIndex+CP_CAN_ADDRESS_CHANNEL_GROUP_LENGTH, CP_CAN_ADDRESS_CHANNEL_ID_SET_LENGTH, pCommNodeFilterList->Filter[i].dwChannelIDSet);
        iIndex+=CP_CAN_ADDRESS_LENGTH;
    }
}
/**
  * @}
  */
