/**
  ******************************************************************************
  * @file    DiscProtocol.c
  * @project CAN-ENET Software Support Package
  *
  * @version V2.0.0
  * @date    April 2021
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * This file provides functions to support messages from the \ref Documentation 
  * "Ethernet to CAN Converter Discovery Protocol. Version: 1A". The protocol 
  * independent message structure is supported in \link PMessage.h PMessage\endlink
  * module.
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2021 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/

#if defined(_WIN32) || defined(_WIN64)
    #define _CRT_SECURE_NO_WARNINGS
#endif

#include <string.h>
#include <assert.h>
#include <stdio.h>

#include "../Inc/DiscProtocol.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @addtogroup DiscProtocol
  * @{
  */

/* Private constants------------------------------------------------------------*/
/** @defgroup DiscProtocol_Private_Constants Private Constants
  * @{
  */
#define DISC_PROTOCOL_ID 1 /**< Discovery Protocol ID. */

/** @defgroup DiscProtocol_Discovery_Response_Constants Response Message Constants
  * @{
  */
//---------------------------------------------------------------------------------------------------
#define DISCOVERY_RESPONSE_MAC_ADDR_INDEX       0
#define DISCOVERY_RESPONSE_MAC_ADDR_LENGTH      6
#define DISCOVERY_RESPONSE_IP_ADDR_INDEX        (DISCOVERY_RESPONSE_MAC_ADDR_INDEX+DISCOVERY_RESPONSE_MAC_ADDR_LENGTH)
#define DISCOVERY_RESPONSE_IP_ADDR_LENGTH       4
#define DISCOVERY_RESPONSE_WEB_PORT_INDEX       (DISCOVERY_RESPONSE_IP_ADDR_INDEX+DISCOVERY_RESPONSE_IP_ADDR_LENGTH)
#define DISCOVERY_RESPONSE_WEB_PORT_LENGTH      2
#define DISCOVERY_RESPONSE_DEVICE_PORT_INDEX    (DISCOVERY_RESPONSE_WEB_PORT_INDEX+DISCOVERY_RESPONSE_WEB_PORT_LENGTH)
#define DISCOVERY_RESPONSE_DEVICE_PORT_LENGTH 2
#define DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_INDEX   (DISCOVERY_RESPONSE_DEVICE_PORT_INDEX+DISCOVERY_RESPONSE_DEVICE_PORT_LENGTH)
#define DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_LENGTH 1
#define DISCOVERY_RESPONSE_PART_NUMBER_INDEX    (DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_INDEX+DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_LENGTH)
#define DISCOVERY_RESPONSE_PART_NUMBER_LENGTH   (DP_PART_NUMBER_STRING_LENGTH_MAX+1)
#define DISCOVERY_RESPONSE_SERIAL_NUMBER_INDEX  (DISCOVERY_RESPONSE_PART_NUMBER_INDEX+DISCOVERY_RESPONSE_PART_NUMBER_LENGTH)
#define DISCOVERY_RESPONSE_SERIAL_NUMBER_LENGTH (DP_SERIAL_NUMBER_STRING_LENGTH_MAX+1)
#define DISCOVERY_RESPONSE_MESSAGE_DATA_LENGTH  (DISCOVERY_RESPONSE_SERIAL_NUMBER_INDEX+DISCOVERY_RESPONSE_SERIAL_NUMBER_LENGTH)
/**
  * @}
  */
/**
  * @}
  */
/* Private typedef -----------------------------------------------------------*/
/** @defgroup DiscProtocol_Private_Types Private Types
  * @{
  */
/**
  * @brief Discovery Protocol Message IDs
  */
typedef enum
{
    MESSAGE_ID_UNDEF=0,
    MESSAGE_ID_REQUEST,
    MESSAGE_ID_RESPONSE
}
MESSAGE_ID;
/**
  * @}
  */
/* Private macros ------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/* Exported variables --------------------------------------------------------*/
/* Imported variables --------------------------------------------------------*/
/* Global function definitions------------------------------------------------*/
/** @addtogroup DiscProtocol_Exported_Functions
  * @{
  */
/**
  * @brief Parses the Discovery Request.
  *
  * @note The parser is called on the protocol message.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  *
  * @retval Returns CT_TRUE if the protocol message contains the Discovery Request.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t DPParseRequest(const PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    if(pPMessage->wProtocolID!=DISC_PROTOCOL_ID) return CT_FALSE;
    return (pPMessage->wMessageID==MESSAGE_ID_REQUEST);
}

/**
  * @brief Generates a Discovery Response Message.
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in] pDiscData: pointer to the discovery data structure.
  *
  * @retval None
  */
void DPGenResponseMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,const DP_DISCOVERY_DATA_t *pDiscData)
{
    int i;

    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=DISC_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_RESPONSE;

    for(i=0;i<DISCOVERY_RESPONSE_MAC_ADDR_LENGTH;i++)
    {
        pPMessage->Data[DISCOVERY_RESPONSE_MAC_ADDR_INDEX+i]=pDiscData->MACAddr.Addr[i];
    }
    for(i=0;i<DISCOVERY_RESPONSE_IP_ADDR_LENGTH;i++)
    {
        pPMessage->Data[DISCOVERY_RESPONSE_IP_ADDR_INDEX+i]=pDiscData->IpAddr.Addr[i];
    }
    PMSetDataToMessage(pPMessage,DISCOVERY_RESPONSE_WEB_PORT_INDEX,DISCOVERY_RESPONSE_WEB_PORT_LENGTH,pDiscData->wWebPort);
    PMSetDataToMessage(pPMessage,DISCOVERY_RESPONSE_DEVICE_PORT_INDEX,DISCOVERY_RESPONSE_DEVICE_PORT_LENGTH,pDiscData->wDevicePort);
    PMSetDataToMessage(pPMessage,DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_INDEX,DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_LENGTH,pDiscData->DevicePortType);
    strncpy((char*)&pPMessage->Data[DISCOVERY_RESPONSE_PART_NUMBER_INDEX],pDiscData->szPartNumber,(DISCOVERY_RESPONSE_PART_NUMBER_LENGTH-1));
    strncpy((char*)&pPMessage->Data[DISCOVERY_RESPONSE_SERIAL_NUMBER_INDEX],pDiscData->szSerialNumber,(DISCOVERY_RESPONSE_SERIAL_NUMBER_LENGTH-1));

    pPMessage->wDataLength=DISCOVERY_RESPONSE_MESSAGE_DATA_LENGTH;
}

/**
  * @brief Generates a Discovery Request Message.
  *
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  *
  * @retval None
  */
void DPGenRequestMessage(PM_PROTOCOL_MESSAGE_t *pPMessage)
{
    memset(pPMessage,0,sizeof(*pPMessage));
    pPMessage->wProtocolID=DISC_PROTOCOL_ID;
    pPMessage->wMessageID=MESSAGE_ID_REQUEST;
}

/**
  * @brief Parses the Discovery Response.
  *
  * @note The parser is called on the protocol message.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[out] pDiscData: pointer to the discovery data structure.
  *
  * @retval Returns CT_TRUE if the protocol message contains the Discovery Response ID
  *         and the protocol message data field has the expected length (can be parsed).
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t DPParseResponse(const PM_PROTOCOL_MESSAGE_t *pPMessage,DP_DISCOVERY_DATA_t *pDiscData)
{
    int i;

    if (pPMessage->wProtocolID != DISC_PROTOCOL_ID) return CT_FALSE;
    if (pPMessage->wMessageID != MESSAGE_ID_RESPONSE) return CT_FALSE;

    if(pPMessage->wDataLength<DISCOVERY_RESPONSE_MESSAGE_DATA_LENGTH) return CT_FALSE;

    memset(pDiscData,0,sizeof(*pDiscData));
    for(i=0;i<DISCOVERY_RESPONSE_MAC_ADDR_LENGTH;i++)
    {
        pDiscData->MACAddr.Addr[i]=pPMessage->Data[DISCOVERY_RESPONSE_MAC_ADDR_INDEX+i];
    }
    for(i=0;i<DISCOVERY_RESPONSE_IP_ADDR_LENGTH;i++)
    {
        pDiscData->IpAddr.Addr[i]=pPMessage->Data[DISCOVERY_RESPONSE_IP_ADDR_INDEX+i];
    }
    pDiscData->wWebPort=PMGetDataFromMessage(pPMessage,DISCOVERY_RESPONSE_WEB_PORT_INDEX,DISCOVERY_RESPONSE_WEB_PORT_LENGTH);
    pDiscData->wDevicePort=PMGetDataFromMessage(pPMessage,DISCOVERY_RESPONSE_DEVICE_PORT_INDEX,DISCOVERY_RESPONSE_DEVICE_PORT_LENGTH);
    pDiscData->DevicePortType=(DP_PORT_TYPE_t)PMGetDataFromMessage(pPMessage,DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_INDEX,DISCOVERY_RESPONSE_DEVICE_PORT_TYPE_LENGTH);
    strncpy(pDiscData->szPartNumber,(char*)&pPMessage->Data[DISCOVERY_RESPONSE_PART_NUMBER_INDEX],(DISCOVERY_RESPONSE_PART_NUMBER_LENGTH-1));
    strncpy(pDiscData->szSerialNumber,(char*)&pPMessage->Data[DISCOVERY_RESPONSE_SERIAL_NUMBER_INDEX],(DISCOVERY_RESPONSE_SERIAL_NUMBER_LENGTH-1));

    return CT_TRUE;
}

/**
  * @brief Compares two discovery data structures.
  *
  * @param[in] pDiscData1: pointer to the first discovery data structure.
  * @param[in] pDiscData2: pointer to the second discovery data structure.
  *
  * @retval Returns CT_TRUE if the discovery data structures contain equal data.
  *         Otherwise, returns CT_FALSE.
  */
BOOL_t DPIsDiscDataEqual(const DP_DISCOVERY_DATA_t *pDiscData1,const DP_DISCOVERY_DATA_t *pDiscData2)
{
    int i;

    for(i=0;i<6;i++)
    {
        if(pDiscData1->MACAddr.Addr[i]!=pDiscData2->MACAddr.Addr[i]) return CT_FALSE;
    }
    for(i=0;i<4;i++)
    {
        if(pDiscData1->IpAddr.Addr[i]!=pDiscData2->IpAddr.Addr[i]) return CT_FALSE;
    }
    if(pDiscData1->wWebPort!=pDiscData2->wWebPort) return CT_FALSE;
    if(pDiscData1->wDevicePort!=pDiscData2->wDevicePort) return CT_FALSE;
    if(pDiscData1->DevicePortType!=pDiscData2->DevicePortType) return CT_FALSE;
    if(strncmp(pDiscData1->szPartNumber,pDiscData2->szPartNumber,DISCOVERY_RESPONSE_PART_NUMBER_LENGTH)!=0) return CT_FALSE;
    if(strncmp(pDiscData1->szSerialNumber,pDiscData2->szSerialNumber,DISCOVERY_RESPONSE_SERIAL_NUMBER_LENGTH)!=0) return CT_FALSE;

    return CT_TRUE;
}

/**
  * @brief Gets a readable name of the Port Type value.
  *
  * @param[in] PortType: Port Type.
  *
  * @retval Returns a pointer to a zero terminated constant character string containing
  *         a readable name of the Port Type value.
  */
const char* DPGetPortTypeName(DP_PORT_TYPE_t PortType)
{
    if(PortType==DP_PORT_TYPE_UDP) return "UDP";
    if(PortType==DP_PORT_TYPE_TCP) return "TCP";
    return "?";
}
/**
  * @}
  */
/* Private function definitions-----------------------------------------------*/
/**
  * @}
  */

/**
  * @}
  */
