/**
  ******************************************************************************
  * @file    HealthData.c
  * @project CAN-ENET Software Support Package
  *
  * @version V4.0.1
  * @date    April 2025
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * This file provides functions for processing of the _Health Data_ field in the
  * _Status Response_ and _Heartbeat_ messages described in the \ref Documentation
  * "Ethernet to CAN Converter Health Status. Version 4". The _Health Data_
  * field contains a set of operational statuses of the converter major hardware
  * and software components together with the aggregated operational status presenting
  * the overall health status of the Ethernet to CAN converter.
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2025 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
/* Includes ------------------------------------------------------------------*/
#include "../Inc/HealthData.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @addtogroup HealthData
  * @{
  */

/* Private constants ---------------------------------------------------------*/
/** @defgroup HealthData_Private_Constants Private Constants
  * @{
  */

/** @defgroup HealthData_Packing_Unpacking_Constants Health Data Packing/Unpacking Constants
  * Defined for two bytes per operational status.
  * @{
  */
#define OPERATIONAL_STATUS_MASK     3
#define OPERATIONAL_STATUS_SHIFT    2
/**
  * @}
  */
/**
  * @}
  */
/* Private typedef -----------------------------------------------------------*/
/* Private macros ------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
/** @defgroup HealthData_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Names of the major hardware and software components in the converter Health Data array.
  */
static const char *gsConverterComponentNames[HD_NUMBER_OF_CONVERTER_COMPONENTS] =
{
    // Should follow the order of OPERATIONAL_STATUS_INDEX
    "Controller Configuration Parameters Loading",
    "CAN Configuration Parameters Loading",
    "Ethernet Configuration Parameters Loading",

    "CAN Driver Initialization",
    "Ethernet Driver Initialization",
    "LwIp Stack Initialization",

    "CAN Operation",
    "Ethernet Operation",

    "Power Supply Voltage",
    "Output CAN Voltage",
    "Internal BAT Voltage",
    "uC Temperature",

    "Flash Memory Driver Initialization",
    "Logging Memory Driver Initialization",
    "CAN Logging Operation"
};
/**
  * @}
  */
/* Exported variables --------------------------------------------------------*/
/* Imported variables --------------------------------------------------------*/
/* Global function definitions------------------------------------------------*/
/** @addtogroup HealthData_Exported_Functions
  * @{
  */
/**
  * @brief Gets a readable name of the Converter Type value.
  *
  * @param[in] ConverterType: Converter Type enumerated value.
  *
  * @retval Returns a pointer to a zero terminated constant character string containing
  *         a readable name of the ConverterType value.
  */
const char *HDGetConverterTypeName(CP_CONVERTER_TYPE_t ConverterType)
{
    const char* cszCAN_ETHERNET="CAN_ETHERNET";
    const char* cszCAN_WIFI="CAN_WIFI";
    const char* cszCAN_WIFI_DATALOG="CAN_WIFI_DATALOG";
    const char* cszGeneric="Generic";

    switch(ConverterType)
    {
        case CP_CONVERTER_CAN_ETHERNET:
            return cszCAN_ETHERNET;

        case CP_CONVERTER_CAN_WIFI:
            return cszCAN_WIFI;

        case CP_CONVERTER_CAN_WIFI_DATALOG:
            return cszCAN_WIFI_DATALOG;
				
				case CP_CONVERTER_GENERIC:
					return cszGeneric;
    }
    return "";
}

/**
  * @brief Gets a readable name of the Operational Status value.
  *
  * @param[in] OperationalStatus: Operational Status enumerated value.
  *
  * @retval Returns a pointer to a zero terminated constant character string containing
  *         a readable name of the OperationalStatus value.
  */
const char *HDGetOperationalStatusName(HD_OPERATIONAL_STATUS_t OperationalStatus)
{
    const char* cszUndef="Undef";
    const char* cszNormal="Normal";
    const char* cszWarning="Warning";
    const char* cszError="Error";

    switch(OperationalStatus)
    {
        case HD_OPERATIONAL_STATUS_UNDEF:
            return cszUndef;

        case HD_OPERATIONAL_STATUS_NORMAL:
            return cszNormal;

        case HD_OPERATIONAL_STATUS_WARNING:
            return cszWarning;

        case HD_OPERATIONAL_STATUS_ERROR:
            return cszError;
    }
    return cszUndef;
}

/**
  * @brief Gets a readable description string for a converter major hardware or software component in the converter
  *        Health Data array.
  *
  * @param[in] ConverterComponent: The converter major hardware or software component enumerated value.
  * @param[in] ConverterType: Converter Type.
  *
  * @retval Returns a pointer to a zero terminated constant character string containing
  *         a readable description of the ConverterComponent value.
  */
const char *HDGetConverterComponentName(HD_CONVERTER_COMPONENT_t ConverterComponent,CP_CONVERTER_TYPE_t ConverterType)
{
    if(HDIsConverterComponentInUse(ConverterComponent,ConverterType))
    {
        if(ConverterType==CP_CONVERTER_CAN_WIFI || ConverterType==CP_CONVERTER_CAN_WIFI_DATALOG)
        {
            if(ConverterComponent==HD_CONVERTER_WIFI_CP_LOAD) return "WiFi Configuration Parameters Loading";
            if(ConverterComponent==HD_CONVERTER_WIFI_DRIVER_INIT) return "WiFi Driver Initialization";
            if(ConverterComponent==HD_CONVERTER_WIFI_OPERATION) return "WiFi Operation";
        }
        if(ConverterType==CP_CONVERTER_GENERIC)
        {
            if(ConverterComponent==HD_CONVERTER_INTERNAL_uC_VOLTAGE) return "Internal uC Voltage";
            if(ConverterComponent==HD_CONVERTER_IP_STACK_INIT) return "IP Stack Initialization";
        }
        if(ConverterComponent<HD_NUMBER_OF_CONVERTER_COMPONENTS) return gsConverterComponentNames[ConverterComponent];
    }
    return "";
}

/**
  * @brief Check whether the Converter Component is in use by the specified Converter Type
  *
  * @param[in] ConverterComponent: Converter Component.
  * @param[in] ConverterType: Converter Type.
  *
  * @retval Returns Boolean value.
  */
BOOL_t HDIsConverterComponentInUse(HD_CONVERTER_COMPONENT_t ConverterComponent,CP_CONVERTER_TYPE_t ConverterType)
{
    if(ConverterComponent>=HD_NUMBER_OF_CONVERTER_COMPONENTS) return CT_FALSE;
        switch(ConverterType)
       {
            case CP_CONVERTER_CAN_ETHERNET:
              if(ConverterComponent==HD_CONVERTER_LOGGING_MEMORY_DRIVER_INIT || ConverterComponent==HD_CONVERTER_CAN_LOGGING_OPERATION) return CT_FALSE;
              break;

            case CP_CONVERTER_CAN_WIFI:
						case CP_CONVERTER_GENERIC:
              if(ConverterComponent==HD_CONVERTER_OUTPUT_CAN_VOLTAGE || ConverterComponent==HD_CONVERTER_LOGGING_MEMORY_DRIVER_INIT || ConverterComponent==HD_CONVERTER_CAN_LOGGING_OPERATION) return CT_FALSE;
              break;

            case CP_CONVERTER_CAN_WIFI_DATALOG:
              if(ConverterComponent==HD_CONVERTER_OUTPUT_CAN_VOLTAGE) return CT_FALSE;
              break;

            default:
              return CT_FALSE;
        }
        return CT_TRUE;
}

/**
  * @brief Packs a Health Data array of the individual major hardware and software component operational statuses into a
  *        4-byte value
  *
  * @param[in] pHealthData: pointer to the Health Data array.
  * @param[in] ConverterType: Converter Type.
  *
  * @retval Returns a 4-byte value of the packed Health Data.
  */
DWORD_t HDPackHealthData(const HD_HEALTH_DATA_t *pHealthData,CP_CONVERTER_TYPE_t ConverterType)
{
    DWORD_t dwHealthData;
    int i;

    dwHealthData=0;
    dwHealthData|=HDCalcConverterHealthStatus(pHealthData,ConverterType) & OPERATIONAL_STATUS_MASK;

    for(i=0;i<HD_NUMBER_OF_CONVERTER_COMPONENTS;i++)
    {
        if(HDIsConverterComponentInUse((HD_CONVERTER_COMPONENT_t)i,ConverterType))
        {
            dwHealthData|=((DWORD_t)pHealthData->OperationalStatus[i] & OPERATIONAL_STATUS_MASK)<<((i+1)*OPERATIONAL_STATUS_SHIFT);
        }
    }
    return dwHealthData;
}

/**
  * @brief Unpacks a 4-byte Health Data received over the Ethernet into a Health Data array of
  *        the individual major hardware and software component operational statuses.
  *
  * @param[in]  dwHealthData: 4-byte packed Health Data.
  * @param[out] pHealthData: pointer to the Health Data array.
  * @param[in] ConverterType: Converter Type.
  *
  * @retval The converter Health Status (the aggregated converter operational status).
  */
HD_OPERATIONAL_STATUS_t HDUnpackHealthData(DWORD_t dwHealthData,HD_HEALTH_DATA_t *pHealthData,CP_CONVERTER_TYPE_t ConverterType)
{
    HD_OPERATIONAL_STATUS_t HealthStatus=(HD_OPERATIONAL_STATUS_t)(dwHealthData & OPERATIONAL_STATUS_MASK);
    int i;

    for(i=0;i<HD_NUMBER_OF_CONVERTER_COMPONENTS;i++)
    {
        if(HDIsConverterComponentInUse((HD_CONVERTER_COMPONENT_t)i,ConverterType))
        {
            pHealthData->OperationalStatus[i]=(HD_OPERATIONAL_STATUS_t)((dwHealthData>>((i+1)*OPERATIONAL_STATUS_SHIFT)) & OPERATIONAL_STATUS_MASK);
        }
    }
    return HealthStatus;
}

/**
  * @brief Calculates the converter Health Status.
  *
  * @note The converter Health Status is an aggregated converter operational status
  *        based on the individual major hardware and software component operational statuses from the Health Data array.
  *
  * @param[in] pHealthData: pointer to the Health Data array.
  * @param[in] ConverterType: Converter Type.
  *
  * @retval The converter Health Status.
  */
HD_OPERATIONAL_STATUS_t HDCalcConverterHealthStatus(const HD_HEALTH_DATA_t *pHealthData,CP_CONVERTER_TYPE_t ConverterType)
{
    int i;

    for(i=0;i<HD_NUMBER_OF_CONVERTER_COMPONENTS;i++)
    {
        if(HDIsConverterComponentInUse((HD_CONVERTER_COMPONENT_t)i,ConverterType))
        {
            if(pHealthData->OperationalStatus[i]==HD_OPERATIONAL_STATUS_ERROR) return HD_OPERATIONAL_STATUS_ERROR;
        }
    }

    for(i=0;i<HD_NUMBER_OF_CONVERTER_COMPONENTS;i++)
    {
        if(HDIsConverterComponentInUse((HD_CONVERTER_COMPONENT_t)i,ConverterType))
        {
            if(pHealthData->OperationalStatus[i]==HD_OPERATIONAL_STATUS_WARNING) return HD_OPERATIONAL_STATUS_WARNING;
        }
    }

    for(i=0;i<HD_NUMBER_OF_CONVERTER_COMPONENTS;i++)
    {
        if(HDIsConverterComponentInUse((HD_CONVERTER_COMPONENT_t)i,ConverterType))
        {
            if(pHealthData->OperationalStatus[i]==HD_OPERATIONAL_STATUS_UNDEF) return HD_OPERATIONAL_STATUS_UNDEF;
        }
    }
    return HD_OPERATIONAL_STATUS_NORMAL;
}

/**
  * @}
  */
/* Private function definitions-----------------------------------------------*/
/**
  * @}
  */

/**
  * @}
  */
