/**
  ******************************************************************************
  * @file    PMessage.c
  * @project CAN-ENET Software Support Package
  *
  * @version V2.0.0
  * @date    April 2021
  * @author  Oleksandr Bogush
  *
  * @copyright
  * (c) Axiomatic Technologies Corp. All rights reserved.
  *
  * @brief
  * This file provides functions to support the protocol independent message 
  * structure described in \ref Documentation 
  * "Ethernet to CAN Converter Communication Protocol. Version: 4".
  *
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2021 Axiomatic Technologies Corporation</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of Axiomatic Technologies Corporation nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */ 
/* Includes ------------------------------------------------------------------*/
#include <string.h>
#include <assert.h>
#include <stdio.h>

#include "../Inc/PMessage.h"

/** @addtogroup CAN_ENET_SSP
  * @{
  */

/** @addtogroup PMessage
  * @{
    */

/* Private constants ---------------------------------------------------------*/
/* Private typedef -----------------------------------------------------------*/
/** @defgroup PMessage_Private_Typedef Private Types
  * @{
  */
/**
  * @brief Parser return value
  */
typedef enum
{
    PARSER_OK = 0,
    PARSER_WRONG_FORMAT,
    PARSER_MESSAGE_IS_READY
}
PARSER_RETURN;
/**
  * @}
  */
/* Private macros ------------------------------------------------------------*/
/* Private function prototypes -----------------------------------------------*/
/** @defgroup PMessage_Private_Functions Private Functions
  * @{
  */
static PARSER_RETURN PMParser(PM_PROTOCOL_PARSER_t *pPParser, BYTE_t bData);
/**
  * @}
  */
/* Private variables ---------------------------------------------------------*/
/** @defgroup PMessage_Private_Variables Private Variables
  * @{
  */
static const BYTE_t gAxioProtHeader[PM_PROTOCOL_HEADER_LENGTH]={0x41,0x58,0x49,0x4F};
/**
  * @}
  */
/* Exported variables --------------------------------------------------------*/
/* Imported variables --------------------------------------------------------*/
/* Global function definitions------------------------------------------------*/
/** @addtogroup PMessage_Exported_Functions
  * @{
  */

/**
  * @brief  Parses the protocol message from the input buffer.
  *
  * @note   The protocol message parser structure pPParser should be initialized with all zeros prior to the first call.
  *         All consequent calls do not require initialization.
  *
  * @param[in]  pBuffer: pointer to the input buffer.
  * @param[in]  iDataLength: input data length.
  * @param[out] pPParser: pointer to the protocol message parser structure.
  * @param[in]  OnDataParsed: callback function called on successful parsing of the
  *             protocol message. The function is provided with a pointer to the parsed
  *             protocol message structure and with the arg pointer.
  * @param[in]  OnDataParsedError: callback function called on parsing error. The function
  *             is called only once per parsing of the input buffer. The function is
  *             provided with the arg pointer.
  * @param[in]  arg: pointer to an arbitrary data passed to OnDataParsed() and OnDataParsedError()
  *             functions.
  *
  * @retval None
  */
void PMParseFromBuffer(const BYTE_t *pBuffer, int iDataLength, PM_PROTOCOL_PARSER_t *pPParser,PM_ON_DATA_PARSED_FUNCTION_t OnDataParsed, PM_ON_DATA_PARSED_ERROR_FUNCTION_t OnDataParsedError, void *arg)
{
    int i;
    BYTE_t bData;
    PARSER_RETURN ParserReturn;
    BOOL_t bIsErrorReported;

    bIsErrorReported=CT_FALSE;
    for (i = 0; i < iDataLength; i++)
    {
        bData = pBuffer[i];
        ParserReturn = PMParser(pPParser, bData);
        if (ParserReturn == PARSER_MESSAGE_IS_READY)
        {
            if(OnDataParsed!=NULL) OnDataParsed(&pPParser->PMessage, arg);
            // Reset parser
            memset(pPParser, 0, sizeof(*pPParser));
        }
        else if (ParserReturn == PARSER_WRONG_FORMAT)
        {
            // No need to reset the parser. It has already been reset in PFParser()
            if(!bIsErrorReported)
            {
                if(OnDataParsedError!=NULL) OnDataParsedError(arg);
                bIsErrorReported=CT_TRUE;
            }
        }
    }
}

/**
  * @brief Copies the protocol message to the output buffer.
  *
  * @param[in]  pPMessage: pointer to the protocol message structure.
  * @param[out] pBuffer: pointer to the output buffer.
  * @param[in]  BufferSize: size of the output buffer.
  * @param[out] piBytesToSend: pointer to variable that receives the number of bytes
  *             copied to the output buffer. If the function fails, this value is set to 0.
  *
  * @retval If the function succeeds, returns CT_TRUE. Otherwise, returns CT_FALSE.
  */
BOOL_t PMCopyToBuffer(const PM_PROTOCOL_MESSAGE_t *pPMessage,BYTE_t *pBuffer,size_t BufferSize,int *piBytesToSend)
{
    int i,j;

    if((((size_t)pPMessage->wDataLength) + PM_PROTOCOL_MESSAGE_DATA_INDEX)>BufferSize)
    {
        assert(0);
        *piBytesToSend=0;
        return CT_FALSE;
    }
    memset(pBuffer,0,BufferSize);
    for(i=0;i<PM_PROTOCOL_HEADER_LENGTH;i++)
    {
        pBuffer[i]=gAxioProtHeader[i];
    }
    pBuffer[i++]=(BYTE_t)pPMessage->wProtocolID;
    pBuffer[i++]=(BYTE_t)(((WORD_t)pPMessage->wProtocolID)>>8);

    pBuffer[i++]=(BYTE_t)pPMessage->wMessageID;
    pBuffer[i++]=(BYTE_t)(((WORD_t)pPMessage->wMessageID)>>8);

    pBuffer[i++]=pPMessage->bMessageVersion;

    pBuffer[i++]=(BYTE_t)pPMessage->wDataLength;
    pBuffer[i++]=(BYTE_t)(((WORD_t)pPMessage->wDataLength)>>8);

    for(j=0;j<pPMessage->wDataLength;j++)
    {
        pBuffer[i++]=pPMessage->Data[j];
    }

    *piBytesToSend=i;
    return CT_TRUE;
}

/**
  * @brief Sets the data in the data field of the protocol message structure.
  *
  * @note Up to 4 bytes can be set, LSB first.
  *
  * @param[out] pPMessage: pointer to the protocol message structure.
  * @param[in]  iIndex: index of the data in the data field.
  * @param[in]  iSize: data size. Can be from 0 to 4.
  * @param[in]  dwData: Data to be set in the data field of the protocol message structure.
  *
  * @retval None
  */
void PMSetDataToMessage(PM_PROTOCOL_MESSAGE_t *pPMessage,int iIndex,int iSize,DWORD_t dwData)
{
    int i;
    for(i=0;i<iSize;i++)
    {
        pPMessage->Data[i+iIndex]=(BYTE_t)(dwData & 0xff);
        dwData>>=8;
    }
}

/**
  * @brief Gets the data from the data field of the protocol message structure.
  *
  * @note Up to 4 bytes can be retrieved, LSB first.
  *
  * @param[in] pPMessage: pointer to the protocol message structure.
  * @param[in]  iIndex: index of the data in the data field of the protocol message structure.
  * @param[in]  iSize: data size. Can be from 0 to 4.
  *
  * @retval The data value.
  */
DWORD_t PMGetDataFromMessage(const PM_PROTOCOL_MESSAGE_t *pPMessage,int iIndex,int iSize)
{
    DWORD_t dwData=0;

    for(int i=0;i<iSize;i++)
    {
        dwData|=((DWORD_t)pPMessage->Data[i+iIndex])<<(i*8);
    }

    return dwData;
}
/**
  * @}
  */

/* Private function definitions-----------------------------------------------*/
/** @addtogroup PMessage_Private_Functions
  * @{
  */

/**
  * @brief Parses a protocol message.
  *
  * @note The parser structure should be initialized with all zeros
  *       prior to the first call.
  *       The parser should be called for every byte in the input data stream.
  *       The parser automatically re-initializes the parser structure
  *       with all zeros on error.
  *
  * @param[in,out] pPParser: pointer to the parser structure.
  * @param[in]  bData: data byte from the input data stream.
  *
  * @retval The result of the parsing operation.
  */
PARSER_RETURN PMParser(PM_PROTOCOL_PARSER_t *pPParser, BYTE_t bData)
{
    PM_PROTOCOL_MESSAGE_t *PMessage;

    PMessage = &pPParser->PMessage;

    switch (pPParser->ParserState)
    {
    case PM_PARSER_STATE_INIT:
        assert(pPParser->iIndex == 0);
        pPParser->iIndex = 0;
        if (bData != gAxioProtHeader[pPParser->iIndex++])
        {
            //Wrong message format. Waiting for the header tag
            pPParser->iIndex = 0;
            return PARSER_WRONG_FORMAT;
        }
        pPParser->ParserState = PM_PARSER_STATE_HEADER;
        return PARSER_OK;

    case PM_PARSER_STATE_HEADER:
        assert(pPParser->iIndex<PM_PROTOCOL_HEADER_LENGTH);
        if (bData != gAxioProtHeader[pPParser->iIndex++])
        {
            // Wrong frame format. Tag has not been recognized
            memset(pPParser, 0, sizeof(*pPParser));
            return PARSER_WRONG_FORMAT;
        }
        if (pPParser->iIndex<PM_PROTOCOL_HEADER_LENGTH) return PARSER_OK;
        pPParser->iIndex = 0;
        pPParser->wData = 0;
        pPParser->ParserState = PM_PARDER_STATE_PROTOCOL_ID;
        return PARSER_OK;

    case PM_PARDER_STATE_PROTOCOL_ID:
        assert(pPParser->iIndex<PM_PROTOCOL_ID_LENGTH);
        pPParser->wData |= ((WORD_t)bData) << (8 * pPParser->iIndex);
        if (++pPParser->iIndex<PM_PROTOCOL_ID_LENGTH) return PARSER_OK;

        PMessage->wProtocolID = pPParser->wData;
        pPParser->iIndex = 0;
        pPParser->wData = 0;
        pPParser->ParserState = PM_PARSER_STATE_MESSAGE_ID;
        return PARSER_OK;

    case PM_PARSER_STATE_MESSAGE_ID:
        assert(pPParser->iIndex<PM_PROTOCOL_MESSAGE_ID_LENGTH);
        pPParser->wData |= ((WORD_t)bData) << (8 * pPParser->iIndex);
        if (++pPParser->iIndex<PM_PROTOCOL_MESSAGE_ID_LENGTH) return PARSER_OK;

        PMessage->wMessageID = pPParser->wData;
        PMessage->bMessageVersion = 0;
        pPParser->ParserState = PM_PARSER_STATE_MESSAGE_VERSION;
        return PARSER_OK;

    case PM_PARSER_STATE_MESSAGE_VERSION:
        PMessage->bMessageVersion = bData;
        pPParser->iIndex = 0;
        PMessage->wDataLength = 0;
        pPParser->ParserState = PM_PARSER_STATE_DATA_LENGTH;
        return PARSER_OK;

    case PM_PARSER_STATE_DATA_LENGTH:
        assert(pPParser->iIndex<PM_PROTOCOL_MESSAGE_DATA_LENGTH_LENGTH);
        PMessage->wDataLength |= ((WORD_t)bData) << (8 * pPParser->iIndex);
        if (++pPParser->iIndex<PM_PROTOCOL_MESSAGE_DATA_LENGTH_LENGTH) return PARSER_OK;
        if (PMessage->wDataLength>PM_PROTOCOL_MESSAGE_DATA_SIZE)
        {
            // Wrong message format. wDataLength > PROTOCOL_MESSAGE_DATA_SIZE
            memset(pPParser, 0, sizeof(*pPParser));
            return PARSER_WRONG_FORMAT;
        }
        pPParser->iIndex = 0;
        pPParser->ParserState = PM_PARSER_STATE_DATA;
        if (PMessage->wDataLength == 0) return PARSER_MESSAGE_IS_READY;
        return PARSER_OK;

    case PM_PARSER_STATE_DATA:
        assert(pPParser->iIndex<sizeof(PMessage->Data));
        PMessage->Data[pPParser->iIndex++] = bData;
        if (pPParser->iIndex<PMessage->wDataLength) return PARSER_OK;
        return PARSER_MESSAGE_IS_READY;
    }
    // Reset parser
    assert(0);
    // Parser Error. Unknown ParserState. Should not happen
    memset(pPParser, 0, sizeof(*pPParser));
    return PARSER_WRONG_FORMAT;
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
